import React from 'react';
import PropTypes from 'prop-types';
import Textinput from '@crm/components/dist/lego2/Textinput';
import debounce from 'lodash/debounce';
import cx from 'classnames';
import api from 'api';
import Menu from './Menu';
import MenuItem from './MenuItem';
import css from './styles.modules.scss';

const SEARCH_STRING_REGEX = /^(.+[;,])?\s*([^;,\s](\s*[^;,\s]*)*)?$/;

const INIT_STATE = {
  selectedId: 0,
  items: [],
};

const DEBOUNCE_DELAY = 200;

export default class EmailInput extends React.PureComponent {
  constructor(props) {
    super(props);
    this.loadSuggest = debounce(this.loadSuggest, DEBOUNCE_DELAY);
    this.state = INIT_STATE;
  }

  componentWillReceiveProps(nextProps) {
    if (nextProps.focused) {
      if (nextProps.value !== this.props.value || (nextProps.focused && !this.props.focused)) {
        this.loadSuggest(nextProps.value);
      }
    } else if (this.props.focused) {
      this.setState(INIT_STATE);
    }
  }

  onHoverIndexChange = (index) => {
    this.setState({ selectedId: index });
  };

  getSearchValue(str) {
    const result = str.match(SEARCH_STRING_REGEX);
    if (result && result.length) {
      return result[2];
    }

    return null;
  }

  loadSuggest(value) {
    const searchValue = this.getSearchValue(value) || '';
    if (searchValue.length >= this.props.minSearchStringLength) {
      this.props
        .provider({ text: searchValue.toLowerCase(), ...this.props.extraFetchParam })
        .then((data) => {
          if (this.props.focused) {
            this.setState({
              selectedId: 0,
              items: data.items,
            });
          }
        });
    } else if (this.state.items.length) {
      this.setState(INIT_STATE);
    }
  }

  handleKeyDown = (e) => {
    if (
      this.menu &&
      (e.key === 'ArrowUp' || e.key === 'ArrowDown' || e.key === 'Enter' || e.key === 'Escape')
    ) {
      if (e.key === 'ArrowUp') {
        this.menu.previous();
      } else if (e.key === 'ArrowDown') {
        this.menu.next();
      } else if (e.key === 'Enter') {
        if (this.state.items[this.state.selectedId]) {
          this.insert(this.state.items[this.state.selectedId].email);
        }
      } else if (e.key === 'Escape') {
        this.setState(INIT_STATE);
      }
      e.preventDefault();
    } else if (e.key === 'Enter') {
      e.preventDefault();
    }
  };

  insert(email) {
    const matchesArray = this.props.value.match(SEARCH_STRING_REGEX);

    let nextValue;
    if (matchesArray && matchesArray[1]) {
      nextValue = `${matchesArray[1]} ${email}`;
    } else {
      nextValue = email;
    }

    nextValue = `${nextValue}; `;

    this.props.onChange(nextValue);
    this.setState(INIT_STATE);
  }

  handleItemClick = (email) => {
    this.insert(email);
  };

  getMenuRef = (node) => {
    this.menu = node;
  };

  render() {
    const { className, ...passThroughProps } = this.props;
    return (
      <div className={cx(css.root, className)}>
        <Textinput hasClear={false} {...passThroughProps} onKeyDown={this.handleKeyDown} />
        {Boolean(this.state.items.length) && (
          <Menu
            ref={this.getMenuRef}
            className={css.suggestContainer}
            hoverIndex={this.state.selectedId}
            onHoverIndexChange={this.onHoverIndexChange}
            items={this.state.items}
            itemComponent={MenuItem}
            itemProps={{
              onClick: this.handleItemClick,
            }}
          />
        )}
      </div>
    );
  }
}

EmailInput.propTypes = {
  className: PropTypes.string,
  provider: PropTypes.func,
  onChange: PropTypes.func.isRequired,
  value: PropTypes.string,
  focused: PropTypes.bool,
  minSearchStringLength: PropTypes.number,
  extraFetchParam: PropTypes.objectOf(PropTypes.any),
  size: PropTypes.string,
};

EmailInput.defaultProps = {
  className: undefined,
  provider: api.view.suggest.email,
  minSearchStringLength: 1,
  value: '',
  items: undefined,
  focused: false,
  isDataLoading: false,
  size: 's',
  extraFetchParam: undefined,
};
