import { makeAutoObservable } from 'mobx';
import Bluebird from 'bluebird';
import { get } from 'api/common';
import { Participant, AvailabilityPeriods, Activity } from './EventWidget.types';
import { DEFAULT_VIEW_RANGE } from './EventWidget.config';

export class EventWidgetWithApiService {
  private apiRequestPromise?: Bluebird<unknown>;
  private providerUrl: string;

  public participants: Participant[];
  public date: Date;

  constructor() {
    makeAutoObservable(this);
  }

  public setUrl = (url: string) => {
    this.providerUrl = url;
  };

  public setDate = (date: Date) => {
    this.date = date;
  };

  private setParticipants = (participants: Participant[]) => {
    this.participants = participants;
  };

  public load = () => {
    if (this.apiRequestPromise) this.apiRequestPromise.cancel();

    const fromDateTime = new Date(this.date);
    fromDateTime.setHours(DEFAULT_VIEW_RANGE.start, 0, 0, 0);

    const toDateTime = new Date(this.date);
    toDateTime.setHours(DEFAULT_VIEW_RANGE.end, 0, 0, 0);

    const data = {
      fromDateTime: fromDateTime.toISOString(),
      toDateTime: toDateTime.toISOString(),
    };

    this.apiRequestPromise = get<AvailabilityPeriods>({ url: this.providerUrl, data }).then(
      (data) => {
        const participants = this.mapAvailabilityPeriodsToParticipants(data);
        this.setParticipants(participants);
      },
    );

    return this.apiRequestPromise;
  };

  private mapAvailabilityPeriodsToParticipants = (data: AvailabilityPeriods) => {
    return data.subjects.map((el) => ({
      name: el.name,
      events: el.activities.map(this.mapActivityToEvent),
    }));
  };

  private mapActivityToEvent = (activity: Activity, id: number) => ({
    id,
    range: {
      start: new Date(activity.fromDateTime),
      end: new Date(activity.toDateTime),
    },
  });
}
