import React from 'react';
import { connect } from 'react-redux';
import throttle from 'lodash/throttle';
import Modal from '@crm/components/dist/lego2/Modal';
import { selectors, actions } from './reduxSlice';
import withTransformMatrix from './withTransformMatrix';
import { FileType, SlideDirection } from './types';
import { FileViewerProps, FileViewerConnectedStateProps } from './FileViewer.types';
import HeaderFactory from './Header';
import Inner from './Inner';
import Controls from './Controls';
import Preview from './Preview';
import css from './styles.modules.scss';

class FileViewer extends React.Component<FileViewerProps> {
  private static KEY_DOWN_THROTTLE_MS = 300;

  private targetRef: React.RefObject<HTMLDivElement>;

  public constructor(props: FileViewerProps) {
    super(props);
    this.targetRef = React.createRef();
  }

  public componentDidUpdate(prevProps: FileViewerProps): void {
    const { slides, slideIndex, visible, showLoader, hideError } = this.props;
    const prevVisible = prevProps.visible;
    const prevSlide = prevProps.slides && prevProps.slides[prevProps.slideIndex];
    const currentSlide = slides && slides[slideIndex];

    if (!prevVisible && visible) {
      document.addEventListener('keydown', this.handleKeyDown);
      document.addEventListener('keyup', this.handleKeyUp);
      document.addEventListener('dragstart', this.disableDocumentDrag);
      hideError();
    }

    if (prevVisible && !visible) {
      document.removeEventListener('keydown', this.handleKeyDown);
      document.removeEventListener('keyup', this.handleKeyUp);
      document.removeEventListener('dragstart', this.disableDocumentDrag);
    }

    if (
      (!prevSlide && currentSlide) ||
      (prevSlide && currentSlide && prevSlide.id !== currentSlide.id) ||
      (!prevVisible && visible && prevSlide && currentSlide && prevSlide.id === currentSlide.id)
    ) {
      this.props.onReset();
      hideError();

      if (
        currentSlide &&
        (currentSlide.type === FileType.Image || currentSlide.type === FileType.Docviewer)
      ) {
        showLoader();
      }
    }
  }

  private handleKeyDown = throttle(
    (event: KeyboardEvent) => {
      switch (event.key) {
        case 'ArrowRight': {
          this.props.slideToRight();
          break;
        }

        case 'ArrowLeft': {
          this.props.slideToLeft();
          break;
        }

        default:
          break;
      }
    },
    FileViewer.KEY_DOWN_THROTTLE_MS,
    { trailing: false },
  );

  // Firefox fix
  private disableDocumentDrag = (event: DragEvent): false => {
    if (event.preventDefault) {
      event.preventDefault();
    }
    return false;
  };

  private handleKeyUp = (): void => {
    this.handleKeyDown.cancel();
  };

  private handlePreviewClick = (slideIndex: number): void => {
    const { slideTo } = this.props;
    if (slideTo) {
      slideTo(slideIndex);
    }
  };

  private renderPreview(): React.ReactNode {
    return this.props.slides.map(
      (slide, index): React.ReactNode => (
        <Preview
          key={slide.id}
          index={index}
          slide={slide}
          active={this.props.slideIndex === index}
          onClick={this.handlePreviewClick}
        />
      ),
    );
  }

  public render(): React.ReactNode {
    const {
      slides,
      slideIndex,
      visible,
      hide,
      transformation,
      isLoading,
      isFailed,
      showError,
      hideLoader,
      containerRef,
      canSlideLeft,
      canSlideRight,
      slideToLeft,
      slideToRight,
      onMouseDown,
      onMouseMove,
      onMouseUp,
      onMouseLeave,
      onWheel,
    } = this.props;

    if (!visible || slideIndex === -1) {
      return null;
    }

    const currentSlide = slides[slideIndex];

    return (
      <Modal
        visible={visible}
        theme="normal"
        targetRef={this.targetRef}
        onOutsideClick={hide}
        hasAnimation={false}
        contentClassName={css.modalContent}
      >
        <div role="presentation" className={css.wrapper} ref={this.targetRef}>
          <HeaderFactory
            slide={currentSlide}
            onZoomOut={this.props.onZoomOut}
            onZoomIn={this.props.onZoomIn}
            onReset={this.props.onReset}
            onRotateLeft={this.props.onRotateLeft}
            onRotateRight={this.props.onRotateRight}
            onCross={this.props.hide}
          />
          <Inner
            slide={currentSlide}
            transformation={transformation}
            isLoading={isLoading}
            isFailed={isFailed}
            showError={showError}
            hideLoader={hideLoader}
            onMouseDown={onMouseDown}
            onMouseMove={onMouseMove}
            onMouseUp={onMouseUp}
            onMouseLeave={onMouseLeave}
            onWheel={onWheel}
            containerRef={containerRef}
            appendAddon={
              slides.length !== 1 && (
                <Controls
                  canSlideLeft={canSlideLeft}
                  canSlideRight={canSlideRight}
                  slideToLeft={slideToLeft}
                  slideToRight={slideToRight}
                />
              )
            }
          />
          <div className={css.previewListScroll}>
            <div className={css.previewList}>{this.renderPreview()}</div>
          </div>
        </div>
      </Modal>
    );
  }
}

const mapState = (state): FileViewerConnectedStateProps => {
  const { isFailed, isLoading, isVisible, slides, slideIndex } = selectors.getState(state);

  return {
    isFailed,
    isLoading,
    visible: isVisible,
    slides,
    slideIndex,
    canSlideLeft: selectors.canSlide(state, SlideDirection.Left),
    canSlideRight: selectors.canSlide(state, SlideDirection.Right),
  };
};

export default connect(mapState, actions)(withTransformMatrix(FileViewer));
