import React, { FC, MouseEvent } from 'react';
import fileViewerAdapter from 'components/FilesList/fileViewerAdapter';
import { useDispatch, useSelector } from 'react-redux';
import { File } from 'components/FilesList/types';
import { actions as FileViewerActions } from 'components/FileViewer';
import { Attachment } from 'types/Attachment';

type OmitOnFileClick<T> = Omit<T, 'onFileClick'>;

type ClickableFilesProps = {
  files: Attachment[];
  onFileClick?: (event: MouseEvent, file: Attachment) => void;
};

export const withFileViewer = <T extends ClickableFilesProps>(
  Component: React.ComponentType<T>,
): FC<OmitOnFileClick<T>> => {
  const WrappedComponent: FC<OmitOnFileClick<T>> = (props: OmitOnFileClick<T>) => {
    const files = props.files;

    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    const testMode = useSelector((state: any) => state.root.info.testMode);

    const dispatch = useDispatch();

    const showFile = (_event: MouseEvent, file: Attachment) => {
      const slides = files.map((file) => fileViewerAdapter({ ...file, testMode } as File));

      dispatch(FileViewerActions.setFiles(slides));
      dispatch(FileViewerActions.slideTo(files.findIndex((savedFile) => savedFile.id === file.id)));
      dispatch(FileViewerActions.show());
    };

    const newProps: T = { ...props, onFileClick: showFile } as T;

    return <Component {...newProps} />;
  };

  WrappedComponent.displayName = 'WrappedComponent-' + Component.displayName;

  return WrappedComponent;
};
