import React from 'react';
import { Point, TranslateMatrix, IdentityMatrix, ScaleMatrix, RotateMatrix } from './Matrix';
import { Matrix } from './types';

interface ComponentProps {
  transformation: React.CSSProperties;
  onWheel: (event: React.WheelEvent) => void;
  onMouseDown: (event: React.MouseEvent) => void;
  onMouseMove: (event: React.MouseEvent) => void;
  onMouseUp: (event: React.MouseEvent) => void;
  onMouseLeave: () => void;
  onRotateRight: () => void;
  onRotateLeft: () => void;
  onZoomIn: () => void;
  onZoomOut: () => void;
  onReset: () => void;
  containerRef: React.RefObject<HTMLDivElement>;
}

export type TransformProps = ComponentProps;

interface State {
  translateAndScaleMatrix: Matrix;
  rotateMatrix: Matrix;
}

const withTransformMatrix = <P extends {}>(
  Component: React.ComponentType<P & ComponentProps>,
): React.ComponentClass<P, State> => {
  class TransformMatrix extends React.Component<P, State> {
    private static readonly MIN_SCALE = 0.5;

    private static readonly MAX_SCALE = 10;

    private static readonly ROTATE_ANGLE_DEGREE_STEP = 90;

    private static readonly SCALE_STEP = 0.2;

    private isMouseDown: boolean;

    private translateStartPoint: Point | null;

    private tmpMatrix: Matrix | null;

    private containerRef = React.createRef<HTMLDivElement>();

    public constructor(props: P) {
      super(props);
      this.state = {
        translateAndScaleMatrix: new IdentityMatrix(),
        rotateMatrix: new IdentityMatrix(),
      };

      this.isMouseDown = false;
      this.translateStartPoint = null;
      this.tmpMatrix = null;
    }

    private getRelativePoint(e: React.MouseEvent) {
      if (!this.containerRef.current) {
        return null;
      }

      const box = this.containerRef.current.getBoundingClientRect();

      return new Point(e.clientX - box.left - box.width / 2, e.clientY - box.height / 2 - box.top);
    }

    private getWorldSpaceCursor(e: React.MouseEvent) {
      const cursorPoint = this.getRelativePoint(e);

      if (!cursorPoint) {
        return null;
      }

      const { translateAndScaleMatrix } = this.state;

      const worldSpaceCursorVector = translateAndScaleMatrix
        .getInverseMatrix()
        .multiply(new Matrix([[cursorPoint.x], [cursorPoint.y], [1]]));

      return new Point(worldSpaceCursorVector.data[0][0], worldSpaceCursorVector.data[1][0]);
    }

    private getNormalizeDeltaScale(sign: number) {
      const { translateAndScaleMatrix } = this.state;

      const currentScale = translateAndScaleMatrix.getScale();

      const deltaScale = 1 + sign * TransformMatrix.SCALE_STEP;

      const normalizeScale = Math.min(
        Math.max(TransformMatrix.MIN_SCALE, currentScale * deltaScale),
        TransformMatrix.MAX_SCALE,
      );

      return normalizeScale / currentScale;
    }

    private rotate = (degree: number): void => {
      const rotateMatrix = new RotateMatrix(degree);
      this.setState((prevState) => ({
        rotateMatrix: prevState.rotateMatrix.multiply(rotateMatrix),
      }));
    };

    private rotateLeft = (): void => this.rotate(TransformMatrix.ROTATE_ANGLE_DEGREE_STEP);

    private rotateRight = (): void => this.rotate(-TransformMatrix.ROTATE_ANGLE_DEGREE_STEP);

    private scale = (sign: number): void => {
      if (this.isMouseDown) {
        return;
      }

      const { translateAndScaleMatrix } = this.state;

      const normalizeDeltaScale = this.getNormalizeDeltaScale(sign);

      const scaleMatrix = new ScaleMatrix(normalizeDeltaScale);
      this.setState({ translateAndScaleMatrix: translateAndScaleMatrix.multiply(scaleMatrix) });
    };

    private zoomIn = (): void => this.scale(1);

    private zoomOut = (): void => this.scale(-1);

    private handleWheel = (event: React.WheelEvent): void => {
      const worldSpaceCursor = this.getWorldSpaceCursor(event);

      if (!worldSpaceCursor) {
        return;
      }

      const { translateAndScaleMatrix } = this.state;

      const normalizeDeltaScale = this.getNormalizeDeltaScale(Math.sign(event.deltaY));

      if (normalizeDeltaScale === 1) {
        return;
      }

      const posTranslate = new TranslateMatrix(worldSpaceCursor.x, worldSpaceCursor.y);
      const negTranslate = new TranslateMatrix(-worldSpaceCursor.x, -worldSpaceCursor.y);

      const transformationMatrix = translateAndScaleMatrix
        .multiply(posTranslate)
        .multiply(new ScaleMatrix(normalizeDeltaScale))
        .multiply(negTranslate);

      this.setState({
        translateAndScaleMatrix: transformationMatrix,
      });
    };

    private handleMouseDown = (event: React.MouseEvent): void => {
      this.isMouseDown = true;
      this.translateStartPoint = new Point(event.clientX, event.clientY);
      this.tmpMatrix = this.state.translateAndScaleMatrix;
    };

    private handleMouseMove = (event: React.MouseEvent): void => {
      if (!this.isMouseDown || !this.translateStartPoint || !this.tmpMatrix) {
        return;
      }

      const startPoint = this.translateStartPoint;
      const deltaPoint = new Point(event.clientX - startPoint.x, event.clientY - startPoint.y);
      const translateMatrix = new TranslateMatrix(deltaPoint.x, deltaPoint.y);
      const matrix = translateMatrix.multiply(this.tmpMatrix);
      this.setState({
        translateAndScaleMatrix: matrix,
      });
    };

    private resetDragNDrop = () => {
      this.isMouseDown = false;
      this.translateStartPoint = null;
      this.tmpMatrix = null;
    };

    private createTransformation = (): React.CSSProperties => {
      const { data: matrix } = this.state.translateAndScaleMatrix.multiply(this.state.rotateMatrix);
      const a = matrix[0][0];
      const b = matrix[0][1];
      const c = matrix[1][0];
      const d = matrix[1][1];
      const x = matrix[0][2];
      const y = matrix[1][2];
      return {
        transform: `matrix(${a}, ${b}, ${c}, ${d}, ${x}, ${y})`,
      };
    };

    private reset = (): void =>
      this.setState({
        translateAndScaleMatrix: new IdentityMatrix(),
        rotateMatrix: new IdentityMatrix(),
      });

    public render = (): React.ReactElement => {
      return (
        <Component
          {...this.props}
          transformation={this.createTransformation()}
          onWheel={this.handleWheel}
          onMouseDown={this.handleMouseDown}
          onMouseMove={this.handleMouseMove}
          onMouseUp={this.resetDragNDrop}
          onMouseLeave={this.resetDragNDrop}
          onRotateLeft={this.rotateLeft}
          onRotateRight={this.rotateRight}
          onZoomIn={this.zoomIn}
          onZoomOut={this.zoomOut}
          onReset={this.reset}
          containerRef={this.containerRef}
        />
      );
    };
  }

  return TransformMatrix;
};

export default withTransformMatrix;
