import React from 'react';
import PropTypes from 'prop-types';
import { DragSource } from 'react-dnd';
import { connect } from 'react-redux';
import get from 'lodash/get';
import { Spin } from '@yandex-lego/components/Spin/desktop/bundle';
import cx from 'classnames';
import { Text } from '@crm/components/dist/lego2/Text';
import { config } from 'services/Config';
import Button from '@crm/components/dist/lego2/Button';
import Icon from '@crm/components/dist/lego2/Icon';
import { REACT_DND_KEY } from 'components/ReactDndModifiedBackend';
import css from './styles.modules.scss';
import { DNDType } from './constants';
import fileViewerAdapter from './fileViewerAdapter';
import getDownloadLink from './getDownloadLink';

const fileSource = {
  beginDrag: (props) => ({
    id: props.id,
    [REACT_DND_KEY]: true,
  }),
  canDrag: (props) => !props.inFileBox,
};

const collect = (dndConnect) => ({
  connectDragSource: dndConnect.dragSource(),
});

class File extends React.Component {
  handleRemove = () => {
    this.props.onRemove(this.props.id);
  };

  handleFileViewerClick = () => {
    const { onFileViewerClick } = this.props;
    if (!onFileViewerClick) {
      return;
    }

    onFileViewerClick(fileViewerAdapter(this.props));
  };

  renderRemove() {
    if (typeof this.props.onRemove === 'function') {
      return (
        <Button
          icon={(cls) => <Icon className={cls} svg="close" />}
          title="Удалить файл"
          view="clear"
          onClick={this.handleRemove}
          disabled={this.props.disabled}
          className={cx(css.File__remove, css.File__item)}
          size=""
        />
      );
    }

    return null;
  }

  render() {
    const {
      size,
      name,
      className,
      isFullName,
      type,
      isLoading,
      forceUseFeatureIssueTimelineV2,
    } = this.props;
    const nameClass = isFullName ? 'crm-fileNameFull' : 'crm-fileName';

    const issueTimelineV2 = forceUseFeatureIssueTimelineV2 || config.value.features.issueTimelineV2;
    const view = this.props.view || (issueTimelineV2 ? 'text' : 'link');

    const renderName = () => {
      const nameElement =
        view === 'text' ? (
          <Text
            title={name}
            typography="body-short-m"
            className={nameClass}
            dangerouslySetInnerHTML={{ __html: name }}
          />
        ) : (
          <span className="crm-item">
            <div className={nameClass} dangerouslySetInnerHTML={{ __html: name }} />
          </span>
        );

      switch (type) {
        case 'Image':
        case 'Docviewer': {
          return (
            <a
              title="Открыть в docviewer"
              className={css.nameLink}
              onClick={this.handleFileViewerClick}
            >
              {nameElement}
            </a>
          );
        }

        default:
          return nameElement;
      }
    };

    if (view === 'text') {
      return this.props.connectDragSource(
        <div className={className}>
          <div className={cx(css.File__container)}>
            {renderName()}
            {size && (
              <Text className={css.File__item} typography="body-short-m" color="secondary">
                {size}
              </Text>
            )}
            <Spin className={css.File__item} progress={isLoading} view="default" size="xxs" />
            {this.renderRemove()}
          </div>
        </div>,
      );
    }

    return this.props.connectDragSource(
      <div className={className}>
        <div className="crm-fileInfo">
          {renderName()}
          {size && <span className="crm-fileSize">&nbsp;{`(${size})`}</span>}
          &nbsp;
          <a
            title="Скачать файл"
            href={getDownloadLink(this.props)}
            className={css.download}
            download
          >
            <i className="fa fa-download" />
          </a>
          {this.renderRemove()}
        </div>
      </div>,
    );
  }
}

File.propTypes = {
  id: PropTypes.number.isRequired,
  name: PropTypes.string.isRequired,
  type: PropTypes.string,
  size: PropTypes.string,
  className: PropTypes.string,
  isFullName: PropTypes.bool,
  isLoading: PropTypes.bool,
  disabled: PropTypes.bool,
  onRemove: PropTypes.func,
  onFileViewerClick: PropTypes.func.isRequired,
  connectDragSource: PropTypes.func.isRequired,
};

File.defaultProps = {
  size: undefined,
  type: undefined,
  className: undefined,
  onRemove: undefined,
  isFullName: true,
  isLoading: false,
  disabled: false,
};

const mapState = (state) => ({
  testMode: get(state, 'root.info.testMode'),
});

export default DragSource(DNDType, fileSource, collect)(connect(mapState)(File));
