import React from 'react';
import { connect } from 'react-redux';
import WordBreak from 'components/WordBreak';
import PropTypes from 'prop-types';
import get from 'lodash/get';
import cx from 'classnames';
import { actions as FileViewerActions } from 'components/FileViewer';
import { DNDType } from './constants';
import fileViewerAdapter from './fileViewerAdapter';
import File from './File';
import css from './styles.modules.scss';
import './style.css';

class FilesList extends React.Component {
  constructor(props) {
    super(props);

    this.state = {
      isMaxHeight: false,
    };

    if (Array.isArray(props.files)) {
      this.files = props.files.map((file) =>
        fileViewerAdapter({
          ...file,
          testMode: this.props.testMode,
        }),
      );
    }
  }

  componentDidMount() {
    this.updateContainer();
  }

  componentDidUpdate(prevProps) {
    const { files } = this.props;
    const prevFiles = prevProps.files;
    if ((prevFiles && prevFiles.length) !== (files && files.length)) {
      this.updateContainer();
    }

    if (prevFiles !== files && Array.isArray(files)) {
      this.files = files.map((file) =>
        fileViewerAdapter({
          ...file,
          testMode: this.props.testMode,
        }),
      );
    }
  }

  showFile = (file) => {
    const { onFileViewerClick } = this.props;
    if (!onFileViewerClick) {
      return;
    }

    onFileViewerClick({
      slides: this.files,
      slideIndex: this.files.findIndex((savedFile) => savedFile.id === file.id),
    });
  };

  updateContainer() {
    if (!this.props.maxHeight) {
      return;
    }

    if (this.container && this.container.clientHeight >= this.props.maxHeight) {
      this.setState({ isMaxHeight: true });
    } else {
      this.setState({ isMaxHeight: false });
    }
  }

  render() {
    const {
      files,
      objectId,
      objectType,
      isInline,
      className,
      rootClassName,
      maxHeight,
      onFileViewerClick,
      ...rest
    } = this.props;

    const itemClass = isInline ? 'crm-filesListItem_inline crm-filesListItem' : 'crm-filesListItem';

    const getDownloadAllLink = () => (
      <div key="download_all" className={itemClass}>
        <div className="crm-fileInfo">
          <a
            href={`${global.CRM_SPACE_API_HOST}/download/${objectType}/files/${objectId}`}
            style={{ fontWeight: 'bold' }}
          >
            <span className="crm-item">
              <div className="crm-downloadAll">Скачать все</div>
            </span>
          </a>
        </div>
      </div>
    );

    if (files && files.length) {
      return (
        <WordBreak
          ref={(node) => {
            this.container = node;
          }}
          className={cx(rootClassName, {
            [className]: this.state.isMaxHeight,
          })}
          style={{
            maxHeight: this.state.isMaxHeight ? maxHeight : undefined,
          }}
        >
          <div className="crm-filesList">
            {objectId && files.length > 1 ? getDownloadAllLink() : ''}
            {files.map((file) => (
              <File
                key={file.id}
                {...file}
                onFileViewerClick={this.showFile}
                className={itemClass}
                {...rest}
              />
            ))}
          </div>
        </WordBreak>
      );
    }

    this.files = [];
    return null;
  }
}

FilesList.propTypes = {
  stKey: PropTypes.string,
  rootClassName: PropTypes.string,
  className: PropTypes.string,
  objectId: PropTypes.number,
  objectType: PropTypes.string,
  maxHeight: PropTypes.number,
  isInline: PropTypes.bool,
  files: PropTypes.arrayOf(PropTypes.shape(File.propTypes)),
  onFileViewerClick: PropTypes.func,
  testMode: PropTypes.bool,
  view: PropTypes.oneOf(['link', 'text']),
};

FilesList.defaultProps = {
  files: [],
  rootClassName: undefined,
  className: css.root,
  stKey: undefined,
  maxHeight: undefined,
  objectId: undefined,
  objectType: 'mail',
  isInline: false,
  onFileViewerClick: undefined,
  testMode: false,
};

export { DNDType };
export default connect(
  (state) => ({
    testMode: get(state, 'root.info.testMode'),
  }),
  (dispatch) => ({
    onFileViewerClick: (payload) => {
      dispatch(FileViewerActions.setFiles(payload.slides));
      dispatch(FileViewerActions.slideTo(payload.slideIndex));
      dispatch(FileViewerActions.show());
    },
  }),
)(FilesList);
