import React from 'react';
import { Field } from 'react-final-form';
import Suggest, { Props as SuggestProps } from 'lego/final-form/Suggest';
import DateTimePicker, { Props as DateTimePickerProps } from 'lego/final-form/DateTimePicker';
import TextInput, { Props as TextInputProps } from 'lego/final-form/TextInput';
import TextArea, { Props as TextAreaProps } from 'lego/final-form/TextArea';
import CheckBox, { Props as CheckBoxProps } from 'lego/final-form/CheckBox';
import Select, { Props as SelectProps } from 'lego/final-form/Select';
import AccountInput, { Props as AccountInputProps } from 'lego/final-form/AccountInput';
import SuggestStaff, { Props as SuggestStaffProps } from 'lego/final-form/SuggestStaff';
import TreeInput, { Props as TreeInputProps } from 'lego/final-form/TreeInput';
import RichTextEditor from 'lego/final-form/RichTextEditor';
import withDictionary, { DictionaryProps } from 'lego/components/withDictionary';
import { Row, Theme } from './Row';
import { Error } from './Error';

export interface FieldRowProps<T, P> {
  className?: string;
  component: T;
  name: string;
  label: React.ReactNode;
  size?: string;
  required?: boolean;
  optional?: boolean;
  inputProps?: P;
  theme?: Theme;
  width?: 'max';
  hasLine?: boolean;
  destroyValueOnUnmount?: boolean;
  [key: string]: unknown;
}

const inputs = {
  Suggest,
  DateTimePicker,
  TextInput,
  TextArea,
  CheckBox,
  Select,
  SelectWithLoad: withDictionary<SelectProps>()(Select),
  AccountInput,
  SuggestStaff,
  TreeInput,
  RichTextEditor,
};

const allowNull = {
  Suggest: true,
  DateTimePicker: false,
  TextInput: false,
  TextArea: false,
  CheckBox: false,
  Select: false,
  SelectWithLoad: false,
  AccountInput: true,
  SuggestStaff: true,
  TreeInput: true,
};

type Props = { fieldClassName?: string } & (
  | FieldRowProps<'Suggest', SuggestProps>
  | FieldRowProps<'DateTimePicker', DateTimePickerProps>
  | FieldRowProps<'CheckBox', CheckBoxProps>
  | FieldRowProps<'Select', SelectProps>
  | (FieldRowProps<'SelectWithLoad', Omit<SelectProps, 'options'>> & DictionaryProps)
  | FieldRowProps<'TextInput', TextInputProps>
  | FieldRowProps<'AccountInput', AccountInputProps>
  | FieldRowProps<'SuggestStaff', SuggestStaffProps>
  | FieldRowProps<'TreeInput', TreeInputProps>
  | FieldRowProps<'TextArea', TextAreaProps>
  | FieldRowProps<'RichTextEditor', {}>
);

export const FieldRow: React.FC<Props> = (props) => {
  const {
    className,
    name,
    component,
    size,
    label,
    required,
    optional,
    theme,
    width,
    fieldClassName,
    hasLine,
    ...other
  } = props;

  const Component = inputs[component];

  if (!Component) {
    return null;
  }

  return (
    <Row
      className={className}
      label={label}
      size={size}
      required={required}
      optional={optional}
      width={width}
      theme={theme}
      hasLine={hasLine}
    >
      <div>
        <Field
          component={Component}
          name={name}
          size={size}
          allowNull={allowNull[component]}
          className={fieldClassName}
          {...other}
        />
      </div>
      <Field name={name}>
        {({ meta }) => (meta.error && meta.touched ? <Error>{meta.error}</Error> : null)}
      </Field>
    </Row>
  );
};
