import React, { isValidElement } from 'react';
import { FormApi } from 'final-form';
import cx from 'classnames';
import { Form as FinalForm, FormProps as FinalFormProps, AnyObject } from 'react-final-form';
import isPromise from 'is-promise';
import Button from '@crm/components/dist/lego2/Button';
import { LegoSizeProp } from '@crm/components/dist/lego2/types';
import { ModalFormProps } from 'lego/utils/modalForm';
import { Schema } from 'yup';
import { yupToFormErrors } from 'utils/yup/yupToFormErrors';
import createI18N from '@yandex-int/i18n';
import * as commonKeyset from 'common.i18n';
import { Title } from '@crm/components/dist/Title';
import Toolbar from 'components/Toolbar';
import css from './Form.module.css';
import { DEFAULT_INPUT_SIZE } from '../constants';
import { ChangedField, FormContext } from './Form.context';

const commonI18n = createI18N(commonKeyset);
const i18nSave = commonI18n('save');
const i18nCancel = commonI18n('cancel');

type FooterAlign = 'start' | 'end';
export interface FormProps<Values = AnyObject> extends ModalFormProps, FinalFormProps<Values> {
  size?: LegoSizeProp;
  title?: React.ReactNode;
  description?: React.ReactNode;
  footerAlign?: FooterAlign;
  successButtonText?: React.ReactNode;
  cancelButtonText?: React.ReactNode;
  validationSchema?: Schema<Values>;
  className?: string;
  onChange?: (field: ChangedField, oldValues: AnyObject) => void;
  buttonsDisabled?: boolean;
}

export class Form<Values = AnyObject> extends React.Component<FormProps<Values>> {
  public static defaultProps = {
    successButtonText: i18nSave,
    cancelButtonText: i18nCancel,
    size: DEFAULT_INPUT_SIZE as LegoSizeProp,
    footerAlign: 'end' as FooterAlign,
  };
  private values: AnyObject;

  private handleSubmit = (values, form) => {
    const result = this.props.onSubmit(values, form);

    if (isPromise(result)) {
      (result as Promise<unknown>).then((data) => {
        this.handleSubmitSuccess(values, form);
        return data;
      });
    } else {
      this.handleSubmitSuccess(values, form);
    }

    return result;
  };

  private handleSubmitSuccess = (values, formikBag) => {
    if (this.props.onSubmitSuccess) {
      this.props.onSubmitSuccess(values, formikBag);
    }
  };

  private handleValidate = (values: Values) => {
    const { validationSchema, validate } = this.props;

    if (validationSchema) {
      return validationSchema
        .validate(values, { abortEarly: false })
        .then(() => ({})) // no error
        .catch((err) => {
          return yupToFormErrors(err);
        });
    }

    if (validate) {
      return validate(values);
    }

    return undefined;
  };

  private renderChildren(form) {
    const { children } = this.props;

    if (isValidElement(children)) {
      return children;
    }

    if (typeof children === 'function') {
      return (children as (form: FormApi<Values>) => React.ReactNode)(form);
    }

    return children;
  }

  private onChange = (value) => {
    if (this.props.onChange) {
      this.props.onChange(value, this.values);
    }
  };

  public render() {
    const {
      size,
      children,
      title,
      description,
      onSubmit,
      successButtonText,
      cancelButtonText,
      validationSchema,
      footerAlign,
      className,
      buttonsDisabled,
      ...other
    } = this.props;

    return (
      <FinalForm onSubmit={this.handleSubmit} validate={this.handleValidate} {...other}>
        {(form) => {
          this.values = form.values;
          return (
            <FormContext.Provider value={{ onChange: this.onChange }}>
              <form className={cx(css.b, className)} onSubmit={form.handleSubmit}>
                {title && <Title className={css.b__title}>{title}</Title>}
                {description && <div className={css.b__description}>{description}</div>}
                {this.renderChildren(form)}
                <Toolbar justifyContent={footerAlign} className={css.b__footer} reverse>
                  <Button
                    size={size}
                    type="submit"
                    view="action"
                    disabled={!form.valid || form.submitting || buttonsDisabled}
                    progress={form.submitting}
                  >
                    {successButtonText}
                  </Button>
                  {this.props.onCancel && (
                    <Button
                      size={size}
                      disabled={form.submitting || buttonsDisabled}
                      onClick={this.props.onCancel}
                    >
                      {cancelButtonText}
                    </Button>
                  )}
                </Toolbar>
              </form>
            </FormContext.Provider>
          );
        }}
      </FinalForm>
    );
  }
}
