import * as React from 'react';
import { FormApi } from 'final-form';
import { FormProps } from 'react-final-form';

interface Options<TValues extends {}> {
  onValuesSave: (values: TValues) => void;
  getSavedValues: () => TValues | undefined | null;
}

export interface WrappedComponentProps<TValues extends {}> {
  initialValues?: TValues;
  onSubmit: FormProps<object>['onSubmit'];
}

export interface OuterWrappedComponentProps {
  filter?: (key: string) => boolean;
}

/**
 * Hoc для сохранения значений формы при последнем сабмите
 * https://st.yandex-team.ru/CRM-9776
 */
export const withSaveValues = <TValues extends {}>(options: Options<TValues>) => <
  TProps extends WrappedComponentProps<TValues>
>(
  WrappedComponent: React.ComponentType<TProps>,
): React.ComponentClass<TProps & OuterWrappedComponentProps> => {
  class SaveValues extends React.Component<TProps & OuterWrappedComponentProps> {
    private initialValues: object | undefined;

    public constructor(props: TProps) {
      super(props);
      this.initialValues = this.getInitialValues();
    }

    private handleSubmit = (values: TValues, form: FormApi): ReturnType<FormProps['onSubmit']> => {
      options.onValuesSave(this.filter(values));
      return this.props.onSubmit(values, form);
    };

    private filter = (values: TValues): TValues => {
      const { filter } = this.props;
      if (filter) {
        const filteredValues = {} as TValues;
        Object.keys(values).forEach((key) => {
          if (filter(key)) {
            filteredValues[key] = values[key];
          }
        });
        return filteredValues;
      }
      return values;
    };

    private getInitialValues = (): TValues => {
      let savedValues = options.getSavedValues();
      if (typeof savedValues !== 'object') {
        savedValues = {} as TValues;
      }
      if (savedValues) {
        savedValues = this.filter(savedValues);
      }

      return {
        ...this.props.initialValues,
        ...savedValues,
      } as TValues;
    };

    public render(): React.ReactNode {
      return (
        <WrappedComponent
          {...this.props}
          initialValues={this.initialValues}
          onSubmit={this.handleSubmit}
        />
      );
    }
  }

  return SaveValues;
};
