import React from 'react';
import PropTypes from 'prop-types';
import { FormattedDate, FormattedNumber } from 'react-intl';
import createI18N, { i18nRaw as createI18NRaw } from '@yandex-int/i18n';
import * as keyset from './FinanceTable.i18n';
import Full from './Full';
import Compact from './Compact';

const i18n = createI18N(keyset);
const i18nRaw = createI18NRaw(keyset);
const i18nDowntime = i18n('downtime');
const i18nChangeForWeek = i18n('changeForWeek');
const i18nChangeForMonth = i18n('changeForMonth');

const dateFormat = value => {
  if (value) {
    return <FormattedDate value={new Date(value)} month="short" />;
  }
  if (value === 0) {
    return '0';
  }

  return '–';
};

const moneyFormat = value => {
  if (value) {
    return <FormattedNumber value={value} style="currency" currency="RUB" />;
  }
  if (value === 0) {
    return '0';
  }

  return '–';
};

const downtimeFormat = value => (value || value === 0 ? value : '–');
const persentFormat = value => (value || value === 0 ? `${value}%` : '–');

const FinanceTable = ({ view, data, sysDates, ...passThroughProps }) => {
  if (!Array.isArray(data)) {
    return null;
  }

  const values = {
    m0Total: {
      title: i18nRaw('totalFor', { date: dateFormat(sysDates.sday01) }),
      format: moneyFormat,
    },
    m1Total: { title: i18nRaw('totalFor', { date: dateFormat(sysDates.m1) }), format: moneyFormat },
    w0Total: {
      title: i18nRaw('spinoffFor', { date: dateFormat(sysDates.f30) }),
      format: moneyFormat,
    },
    f30Total: {
      title: i18nRaw('forecastFor', { date: dateFormat(sysDates.f30) }),
      format: moneyFormat,
    },
    downtime: { title: i18nDowntime, format: downtimeFormat },
    w12Pct: { title: i18nChangeForWeek, format: persentFormat },
    m01Pct: { title: i18nChangeForMonth, format: persentFormat },
    m0Tac: { title: i18nRaw('tacFor', { date: dateFormat(sysDates.sday01) }), format: moneyFormat },
    m1Tac: { title: i18nRaw('tacFor', { date: dateFormat(sysDates.m1) }), format: moneyFormat },
  };

  return view === 'full' ? (
    <Full data={data} values={values} {...passThroughProps} />
  ) : (
    <Compact data={data} values={values} {...passThroughProps} />
  );
};

FinanceTable.propsTypes = {
  view: PropTypes.oneOf(['full', 'compact']),
};

FinanceTable.defaultProps = {
  view: 'full',
};

export default FinanceTable;
