import React, { Component, RefObject, createRef } from 'react';
import cx from 'classnames';
import { Timeout } from 'utils/Timeout';
import { injectIntl } from 'react-intl';
import { formatTimelineDate } from 'modules/timeline/formatTimelineDate';
import createI18N from '@yandex-int/i18n';
import * as commonKeyset from 'common.i18n';
import { WrappedComponentProps } from 'react-intl';
import { FloatingDateProps, FloatingDateState } from './FloatingDate.types';
import { VISIBILITY_DURATION, HEIGHT, MAX_SMOOTH_UPDATE_SPEED } from './FloatingDate.config';
import css from './FloatingDate.module.css';
import { nearby } from './FloatingDate.utils';

const i18n = createI18N(commonKeyset);
const i18nToday = i18n('today');

type FloatingDateInnerProps = FloatingDateProps & WrappedComponentProps;

class FloatingDateComponent extends Component<FloatingDateInnerProps, FloatingDateState>
  implements FloatingDateInstance {
  private hideTimeout: Timeout;

  private dateCache: Map<string, string> = new Map();

  private containerRef: RefObject<HTMLDivElement> = createRef();

  private overlayRef: RefObject<HTMLDivElement> = createRef();

  private dateRef: RefObject<HTMLDivElement> = createRef();

  private nextDateRef: RefObject<HTMLDivElement> = createRef();

  private lastBottomOffset: [number, number, number, number] = [0, 0, 0, 0];

  private smoothUpdateInProgress: boolean = false;

  public constructor(props: FloatingDateInnerProps) {
    super(props);
    this.state = {
      date: '',
      nextDate: '',
      isVisible: false,
    };

    this.hide = this.hide.bind(this);
    this.handleScroll = this.handleScroll.bind(this);
    this.hideTimeout = new Timeout(this.hide, VISIBILITY_DURATION);
  }

  private display() {
    if (this.state.isVisible) {
      return;
    }

    this.setState({
      isVisible: true,
    });
  }

  private hide() {
    if (!this.state.isVisible) {
      return;
    }

    this.setState({
      isVisible: false,
    });
  }

  private getDate(date?: string): string {
    if (!date) {
      return '';
    }

    if (!this.dateCache.has(date)) {
      if (this.props.formatDate) {
        this.dateCache.set(date, this.props.formatDate(date));
      } else {
        this.dateCache.set(date, formatTimelineDate(date, this.props.intl.formatDate, i18nToday));
      }
    }

    return this.dateCache.get(date)!;
  }

  private clearSmoothUpdateStyles() {
    const containerNode = this.containerRef.current;
    const overlayNode = this.overlayRef.current;
    const dateNode = this.dateRef.current;
    const nextDateNode = this.nextDateRef.current;

    dateNode && (dateNode.style.opacity = '1');
    nextDateNode && (nextDateNode.style.opacity = '0');
    overlayNode && (overlayNode.style.boxShadow = '');
    containerNode && (containerNode.style.transform = 'translate3d(0, 0, 0)');

    this.smoothUpdateInProgress = false;
  }

  private applySmoothUpdateStyles(payload: { overlay: boolean; offset: number }) {
    const { overlay, offset } = payload;

    const dateNode = this.dateRef.current;
    const nextDateNode = this.nextDateRef.current;
    const containerNode = this.containerRef.current;
    const overlayNode = this.overlayRef.current;

    if (overlay) {
      overlayNode && (overlayNode.style.boxShadow = 'inset 0 20px currentColor');
    }

    if (offset) {
      dateNode && (dateNode.style.opacity = `${1 - offset / HEIGHT}`);
      nextDateNode && (nextDateNode.style.opacity = `${offset / HEIGHT}`);
      containerNode && (containerNode.style.transform = `translate3d(0, -${offset}px, 0)`);
    }

    this.smoothUpdateInProgress = true;
  }

  private canSmoothUpdate(bottomOffset: number): boolean {
    this.lastBottomOffset.pop();
    this.lastBottomOffset.unshift(bottomOffset);

    return nearby(this.lastBottomOffset).some(
      (entry) => Math.abs(entry[1] - entry[0]) <= MAX_SMOOTH_UPDATE_SPEED,
    );
  }

  private smoothUpdate(payload: {
    firstVisibleDate: string;
    prevVisibleDate?: string;
    nextVisibleDate?: string;
    firstVisibleHeight: number;
    firstHiddenHeight: number;
  }) {
    const {
      firstVisibleDate,
      prevVisibleDate,
      nextVisibleDate,
      firstHiddenHeight,
      firstVisibleHeight,
    } = payload;

    const date = this.getDate(firstVisibleDate);
    const prevDate = this.getDate(prevVisibleDate);
    const nextDate = this.getDate(nextVisibleDate);

    if (this.state.date !== date) {
      this.clearSmoothUpdateStyles();
      this.setState({ date, nextDate: '' });
    }

    if (date !== prevDate && firstHiddenHeight <= HEIGHT) {
      this.applySmoothUpdateStyles({
        overlay: true,
        offset: 0,
      });
    } else if (date !== nextDate && firstVisibleHeight <= HEIGHT) {
      this.applySmoothUpdateStyles({
        overlay: false,
        offset: HEIGHT - firstVisibleHeight,
      });
      this.setState({ date, nextDate });
    } else if (this.smoothUpdateInProgress) {
      this.clearSmoothUpdateStyles();
    }
  }

  private fastUpdate(firstVisibleDate: string) {
    const date = this.getDate(firstVisibleDate);

    if (this.smoothUpdateInProgress) {
      this.clearSmoothUpdateStyles();
    }

    if (this.state.date !== date) {
      this.setState({ date, nextDate: '' });
    }
  }

  public handleScroll(payload: {
    firstVisibleDate: string;
    prevVisibleDate?: string;
    nextVisibleDate?: string;
    firstVisibleHeight: number;
    firstHiddenHeight: number;
    bottomOffset: number;
  }) {
    if (this.canSmoothUpdate(payload.bottomOffset)) {
      this.smoothUpdate(payload);
    } else {
      this.fastUpdate(payload.firstVisibleDate);
    }

    if (payload.firstVisibleDate) {
      this.display();
      this.hideTimeout.reset();
    }
  }

  public componentWillUnmount() {
    this.hideTimeout.clear();
  }

  public render() {
    const { date, nextDate, isVisible } = this.state;

    const className = cx(css.FloatingDate, {
      [css.FloatingDate_visible]: isVisible,
    });

    return (
      <div className={className}>
        <div className={css.FloatingDate__container} ref={this.containerRef}>
          <div className={css.FloatingDate__overlay} ref={this.overlayRef}>
            <div className={css.FloatingDate__date} ref={this.dateRef}>
              {date}
            </div>
          </div>
          <div className={css.FloatingDate__nextDate} ref={this.nextDateRef}>
            {nextDate}
          </div>
        </div>
      </div>
    );
  }
}

export type FloatingDateInstance = Pick<FloatingDateComponent, 'handleScroll'>;

export const FloatingDate = injectIntl(FloatingDateComponent, {
  forwardRef: true,
}) as React.ComponentType<FloatingDateProps & React.RefAttributes<FloatingDateInstance>>;
