import React, { FC, useEffect, useRef, useMemo } from 'react';
import FocusLock from 'react-focus-lock';
import { observer } from 'mobx-react-lite';
import { useFocus } from 'utils/hooks/useFocus';
import { Provider as FocusScopeProvider, useFocusScope } from './FocusScopeContext';
import { FocusScopeProps } from './FocusScope.types';
import { FocusScopeService } from './FocusScopeService';

export const FocusScope: FC<FocusScopeProps> = observer((props) => {
  const { children, className, canListenKeyboard = true, ...restProps } = props;
  const parentScope = useFocusScope();
  const scope = useMemo(() => new FocusScopeService(parentScope), [parentScope]);

  const registeredIndex = useRef<number | null>(null);
  const rootRef = useRef<HTMLDivElement>(null);
  const isFocused = useFocus(rootRef);

  useEffect(() => {
    if (!parentScope || !isFocused || registeredIndex.current == null) {
      return;
    }

    parentScope.setLastFocusedIndex(registeredIndex.current);
  }, [isFocused, parentScope]);

  useEffect(() => {
    const handleKeyDown = (event: KeyboardEvent) => {
      if (!canListenKeyboard) {
        return;
      }

      if (event.key === 'Enter' && !scope.isWithin()) {
        scope.enter();
        scope.focusRelevantNode();
      }

      if (event.key === 'Escape' && scope.isWithin()) {
        scope.leave('keyboard');
        scope.focusScopeNode();
      }
    };

    const rootNode = rootRef.current!;
    rootNode.addEventListener('keydown', handleKeyDown, true);
    return () => {
      rootNode.removeEventListener('keydown', handleKeyDown, true);
    };
  }, [canListenKeyboard]);

  useEffect(() => {
    const rootNode = rootRef.current!;
    scope.registerScopeNode(rootNode);

    if (parentScope) {
      registeredIndex.current = parentScope.registerNode(rootNode);
    }

    const handleFocusOut = (event: FocusEvent) => {
      const relatedTarget = event.relatedTarget as HTMLElement;
      if (
        !relatedTarget ||
        !rootNode.contains(relatedTarget) ||
        scope.closestScopeToNode(relatedTarget) !== rootNode
      ) {
        scope.leave('keyboard');
      }
    };

    const handleFocusIn = (event: FocusEvent) => {
      const target = event.target as HTMLElement;
      if (
        target &&
        target !== rootNode &&
        rootNode.contains(target) &&
        scope.closestScopeToNode(target) === rootNode
      ) {
        scope.enter();
      }
    };

    const handleDocumentMouseDown = (event: MouseEvent) => {
      if (!event.target || rootNode.contains(event.target as HTMLElement)) {
        return;
      }

      scope.leave();
    };

    rootNode.addEventListener('focusout', handleFocusOut);
    rootNode.addEventListener('focusin', handleFocusIn);
    document.addEventListener('mousedown', handleDocumentMouseDown);
    return () => {
      rootNode.removeEventListener('focusout', handleFocusOut);
      rootNode.removeEventListener('focusin', handleFocusIn);
      document.removeEventListener('mousedown', handleDocumentMouseDown);
      if (parentScope) {
        parentScope.unregisterNode(registeredIndex.current!);
      }
    };
  }, []);

  return (
    <FocusScopeProvider value={scope}>
      <div ref={rootRef} tabIndex={scope.scopeTabIndex} className={className} {...restProps}>
        <FocusLock disabled={!scope.isWithin()}>{children}</FocusLock>
      </div>
    </FocusScopeProvider>
  );
});
