import React from 'react';
import { connect } from 'react-redux';
import cx from 'classnames';
import { toggle } from 'modules/notificationsUI/actions';
import { getOpenState } from 'modules/notificationsUI/selectors';
import css from './Footer.module.css';

interface ConnectedState {
  info: any;
  frontendVersion: string;
  isNotificationsOpen: boolean;
}

interface ConnectedDispatch {
  toggleNotifications: () => void;
}

interface ConnectedProps {
  className?: string;
}

type Props = ConnectedState & ConnectedDispatch & ConnectedProps;

class Footer extends React.Component<Props> {
  private static handleRefreshClick() {
    // eslint-disable-next-line no-restricted-globals
    const result = confirm('При обновлении все несохраненные данные будут утеряны. Обновить?');
    if (result) {
      window.location.reload();
    }
  }

  public render() {
    const {
      info,
      frontendVersion,
      toggleNotifications,
      isNotificationsOpen,
      className,
    } = this.props;

    const { version, testMode } = info;

    const notificationButtonClass = isNotificationsOpen
      ? cx(css.notifications, css.notifications_open)
      : css.notifications;

    return (
      <footer className={cx(css.root, { [css.root_test]: testMode }, className)}>
        {testMode && <span className={css.test}>Тест</span>}
        <span className={css.version}>
          <span className={css.version__label}>Версия&nbsp;</span>
          <span className={css.version__value}>{`${__APP_VERSION__} – ${version}`}</span>
        </span>
        {frontendVersion !== undefined && frontendVersion !== __APP_VERSION__ && (
          <span className={css.newVersion}>
            <span className={css.message}>
              Доступна новая версия CRM. Пожалуйста, обновите страницу.
            </span>
            <button type="button" className={css.refreshButton} onClick={Footer.handleRefreshClick}>
              Обновить
            </button>
          </span>
        )}
        <span className={css.copyRight}>
          <span className={css.copyRight__main}>Коммерческая тайна ООО &quot;Яндекс&quot;</span>
          <span className={css.copyRight__address}>
            , 119021, Россия, г. Москва, ул. Льва Толстого, д. 16
          </span>
        </span>
        <button type="button" className={notificationButtonClass} onClick={toggleNotifications}>
          <span className={css.notifications__text}>Оповещения</span>
          <i className="fa fa-bars" aria-hidden="true" />
        </button>
      </footer>
    );
  }
}

const mapState = state => ({
  info: state.root.info,
  frontendVersion: state.root.frontendVersion,
  isNotificationsOpen: getOpenState(state),
});

const mapDispatch = dispatch => ({
  toggleNotifications: () => dispatch(toggle()),
});

export default connect<ConnectedState, ConnectedDispatch, ConnectedProps>(
  mapState,
  mapDispatch,
)(Footer);
