import React from 'react';
import { Field } from 'react-final-form';
import { FormTextAlign } from 'lego/components/FormTextAlign';
import Suggest from 'lego/final-form/Suggest';
import SuggestGrid from 'lego/final-form/SuggestGrid';
import TreeInput from 'lego/final-form/TreeInput';
import { SuggestField } from 'types/api/form/Fields/Suggest';
import { SuggestArrayField } from 'types/api/form/Fields/SuggestArray';
import { ArrayPreviewDictionary } from 'components/PreviewValue/PreviewDictionary';
import { getStringOrUndefined } from '../../utils/getStringOrUndefined';

const suggestComponentTypes = {
  Tree: TreeInput,
  Flat: Suggest,
  Grid: SuggestGrid,
};

const prepareValue = (value, suggestType) => {
  if (!value) {
    return value;
  }
  if (Array.isArray(value)) {
    return value.map((item) => {
      const newItem = { ...item, id: getStringOrUndefined(item.id) };
      if (suggestType === 'Tree') {
        newItem.fullPath = newItem.name;
      }
      return newItem;
    });
  }
  const newItem = { ...value, id: getStringOrUndefined(value.id) };
  if (suggestType === 'Tree') {
    newItem.fullPath = newItem.name;
  }
  return newItem;
};

const getInitialValue = (field: (SuggestField | SuggestArrayField)['merged']) => {
  const suggestType = field?.options?.suggestType || 'Flat';
  return prepareValue(field.data?.value, suggestType);
};

export const createSuggestAdapter = ({ isSingleValue }: { isSingleValue: boolean }) => {
  return {
    previewComponent(props) {
      return (
        <FormTextAlign as="div" size={props.size}>
          <ArrayPreviewDictionary value={props.field.data?.value} />
        </FormTextAlign>
      );
    },
    editComponent(props) {
      const suggestType = props.field?.options?.suggestType || 'Flat';
      const FieldComponent = suggestComponentTypes[suggestType];
      return (
        <Field
          showOnFocus
          name={props.field.type + props.field.id}
          component={FieldComponent}
          initialValue={getInitialValue(props.field)}
          getKey={(item) => getStringOrUndefined(item.id)}
          size={props.size}
          provider={props.field.provider}
          endpoint={props.field.provider}
          isSingleValue={isSingleValue}
          comboInputField="fullPath"
        />
      );
    },
    getInitialValue,
  };
};
