import { makeAutoObservable } from 'mobx';
import { FormBySchemeProps } from '../../FormByScheme/FormByScheme.types';

type Scheme = FormBySchemeProps['scheme'];
type OnSubmit = FormBySchemeProps['onSubmit'];
type OnChange = FormBySchemeProps['onChange'];

export class OptionalSending {
  public selectedFields: Set<string>;
  private originalSubmit: OnSubmit | undefined;
  private originalChange: OnChange | undefined;
  private scheme: Scheme;

  constructor(scheme: Scheme, originalSubmit?: OnSubmit, originalChange?: OnChange) {
    this.originalSubmit = originalSubmit;
    this.originalChange = originalChange;
    this.scheme = scheme;
    this.selectedFields = new Set();
    makeAutoObservable(this, undefined, { autoBind: true });
  }

  public toggleFieldSelecting(fieldId: string) {
    if (this.selectedFields?.has(fieldId)) {
      this.selectedFields.delete(fieldId);
    } else {
      this.selectedFields?.add(fieldId);
    }
  }

  public isFieldSelected(id: string) {
    return Boolean(this.selectedFields?.has(id));
  }

  public onChange(field, values) {
    if (!this.isFieldSelected(field.name)) {
      this.selectedFields.add(field.name);
    }
    this.originalChange?.(field, values);
  }

  public onSubmit(...args) {
    const [values, ...other] = args;
    const filteredValues = {};
    this.scheme.meta.fields
      .filter((field) => this.scheme.meta.fieldsVisibility.includes(field.id))
      .forEach((field) => {
        const key = field.type + field.id;
        const value = { optionalSendingField: true, value: values[key], needSend: false };
        value.needSend = this.selectedFields.has(key);
        filteredValues[key] = value;
      });
    this.originalSubmit?.call(null, filteredValues, [...other]);
  }
}
