import { makeAutoObservable } from 'mobx';
import Bluebird from 'bluebird';
import { Form, DataSet } from 'types/api/form/Form';
import { get, jsonApiCall } from 'api/common';
import Access from 'utils/Access';

export class FormBySchemeLoadableService {
  public loadError?: Error;
  public formData?: Form = undefined;
  public canEditForm = false;
  private apiRequestPromise?: Bluebird<{}>;

  constructor(private url: string) {
    makeAutoObservable(this);
  }

  loadForm() {
    return (this.apiRequestPromise = get<Form>({ url: this.url, global: false })
      .then((data) => {
        this.setFormData(data);
        return data;
      })
      .catch((error) => {
        this.setLoadError(error);
        throw error;
      }));
  }

  saveForm(data: Omit<DataSet, 'id'> & { id?: string }) {
    this.apiRequestPromise = jsonApiCall({ url: this.url, data, global: false }).then((data) => {
      this.updateFormData(data);
      return data;
    });

    return this.apiRequestPromise;
  }

  destroy() {
    if (this.apiRequestPromise) {
      this.apiRequestPromise.cancel();
    }
  }

  private setLoadError(error: Error) {
    this.loadError = error;
  }

  private setFormData(data: Form) {
    this.formData = data;
    this.canEditForm = data.meta.fields.reduce(
      (ac, field) => ac || Access.isEdit(field.access),
      false,
    );
  }

  private updateFormData(data: DataSet) {
    if (this.formData) {
      this.formData.data[0] = data;
    }
  }
}
