import React, { useState, useEffect, useCallback } from 'react';
import { observer } from 'mobx-react-lite';
import { ErrorBoundary } from 'components/ErrorBoundary';
import { ErrorBlock } from 'components/ErrorBlock';
import { FormBySchemeToggleable, formatFormValuesToDataSet } from 'components/FormByScheme';
import { FormBySchemeLoadableProps } from './FormBySchemeLoadable.types';
import { FormBySchemeLoadableService } from './FormBySchemeLoadable.service';

export const FormBySchemeLoadable = observer((props: FormBySchemeLoadableProps) => {
  const { url, canEdit = true, ...otherProps } = props;
  const [formBySchemeLoadableService] = useState(() => new FormBySchemeLoadableService(url));

  useEffect(() => {
    formBySchemeLoadableService.loadForm();

    return () => {
      formBySchemeLoadableService.destroy();
    };
  }, [formBySchemeLoadableService]);

  const formData = formBySchemeLoadableService.formData;
  const loadError = formBySchemeLoadableService.loadError;

  const handleSubmit = useCallback(
    (values) => {
      let rowId;
      if (formData) {
        rowId = formData.data[0].id;
        return formBySchemeLoadableService.saveForm(
          formatFormValuesToDataSet(values, formData, rowId),
        );
      }
    },
    [formBySchemeLoadableService, formData],
  );

  if (loadError) {
    return <ErrorBlock error={loadError} />;
  }

  if (!formData) {
    return null;
  }

  return (
    <ErrorBoundary>
      <FormBySchemeToggleable
        scheme={formData}
        onSubmit={handleSubmit}
        canEdit={formBySchemeLoadableService.canEditForm && canEdit}
        {...otherProps}
      />
    </ErrorBoundary>
  );
});
