import * as React from 'react';
import cx from 'classnames';
import Button from '@crm/components/dist/lego2/Button';
import Icon from '@crm/components/dist/lego2/Icon';
import css from './HeightLimiter.modules.scss';

interface Props {
  children: React.ReactNode;
  className: string;
  maxHeight: number;
}

interface State {
  show: boolean;
  limited: boolean;
}

export default class HeightLimiter extends React.Component<Props, State> {
  public static defaultProps = {
    maxHeight: 200,
    className: undefined,
  };

  private node: Element | null;

  public constructor(props: Props) {
    super(props);

    this.state = {
      show: false,
      limited: false,
    };
  }

  public componentDidMount(): void {
    this.update();
  }

  public componentDidUpdate(): void {
    this.update();
  }

  private toggle = (): void => {
    this.setState((prevState) => ({
      show: !prevState.show,
    }));
  };

  private update(): void {
    if (this.node && this.node.getBoundingClientRect().height > this.props.maxHeight) {
      if (!this.state.limited) {
        this.setState({ limited: true });
      }
    } else if (this.state.limited) {
      this.setState({ limited: false });
    }
  }

  public render(): React.ReactNode {
    const { children, className } = this.props;
    const { show, limited } = this.state;

    if (!limited) {
      return (
        <div
          className={className}
          ref={(node) => {
            this.node = node;
          }}
        >
          {children}
        </div>
      );
    }

    return (
      <div
        className={cx(css.HeightLimiter, className, {
          [css.HeightLimiter_limited]: limited,
          [css.HeightLimiter_border]: true,
        })}
      >
        <Button
          onClick={this.toggle}
          size="xs"
          className={css.HeightLimiter__toggle}
          pin="brick-brick"
          icon={(iconCls) => (
            <Icon className={iconCls} type="arrow" direction={show ? 'top' : 'bottom'} />
          )}
        />
        <div
          className={cx(css.HeightLimiter__content, {
            [css.HeightLimiter__content_limited]: !show,
          })}
        >
          <div
            ref={(node) => {
              this.node = node;
            }}
          >
            {children}
          </div>
        </div>
      </div>
    );
  }
}
