import React, { CSSProperties } from 'react';
import throttle from 'lodash/throttle';
import Button from '@crm/components/dist/lego2/Button';
import cx from 'classnames';
import Spinner from 'components/Spinner';
import css from './InfiniteList.module.css';
import { GetScrollNode, ScrollNodeType } from '../types';

export interface ScrollData {
  scrollHeight: number;
  clientHeight: number;
  scrollTop: number;
  node: ScrollNodeType;
}

export interface InfiniteListProps extends Omit<React.HTMLAttributes<ScrollNodeType>, 'onScroll'> {
  className?: string;
  classNameScrollNode?: string;
  isEof?: boolean;
  isError?: boolean;
  inject?: boolean;
  isLoading?: boolean;
  isLoadingFirstPage?: boolean;
  header?: React.ReactNode;
  children?: React.ReactNode;
  onScroll?: (scrollData: ScrollData) => void;
  onLoad: () => void;
  loadBeginOffset: number;
  getScrollNode?: GetScrollNode;
  emptyComponent?: React.ReactNode | React.ComponentType<{}>;
  offset?: number;
}

class InfiniteList extends React.Component<InfiniteListProps> {
  protected static spinnerWithWrap = (
    <div className={css.spinnerWrap}>
      <Spinner visible modal={false} />
    </div>
  );

  private scrollNode: HTMLDivElement;

  public static defaultProps = {
    isEof: true,
    isError: false,
    inject: false,
    isLoading: false,
    isLoadingFirstPage: false,
    loadBeginOffset: 200,
  };

  protected handleScroll = throttle(() => {
    this.loadList();

    if (this.props.onScroll) {
      this.props.onScroll(this.getScrollParam());
    }
  }, 300);

  public componentDidMount() {
    this.scrollNode.addEventListener('scroll', this.handleScroll);
  }

  public componentDidUpdate(prevProps) {
    // пробуем подгрузить список
    // например: когда eof===false, а прокрутки нет
    if (prevProps.isLoading && !this.props.isLoading) {
      this.loadList();
    }

    if (this.props.offset === 0 && prevProps.offset !== this.props.offset) {
      this.resetScroll();
    }
  }

  public componentWillUnmount() {
    this.handleScroll.cancel();
    this.scrollNode.removeEventListener('scroll', this.handleScroll);
  }

  private getScrollParam() {
    return {
      scrollHeight: this.scrollNode.scrollHeight,
      clientHeight: this.scrollNode.clientHeight,
      scrollTop: this.scrollNode.scrollTop,
      node: this.scrollNode,
    };
  }

  protected getScrollNode = (node) => {
    const { getScrollNode } = this.props;

    this.scrollNode = node;
    if (typeof getScrollNode === 'function') {
      getScrollNode(node);
    }
  };

  private handleLoad = () => {
    this.props.onLoad();
  };

  public resetScroll = () => {
    this.scrollNode.scrollTop = 0;
  };

  private loadList() {
    const args = this.getScrollParam();

    /*
     * args.clientHeight && ... fast fix for https://st.yandex-team.ru/CRM-9053
     * element with { display: none } has scrollHeight, clientHeight, scrollTop === 0
     * */
    const isInfiniteArea =
      args.clientHeight &&
      args.scrollHeight - args.clientHeight - args.scrollTop < this.props.loadBeginOffset;

    if (!this.props.isLoading && !this.props.isError && !this.props.isEof && isInfiniteArea) {
      this.props.onLoad();
    }
  }

  private renderEmptyPlaceholder() {
    const { emptyComponent } = this.props;

    if (typeof emptyComponent === 'function') {
      const EmptyComponent = emptyComponent as React.ComponentType<{}>;
      return <EmptyComponent />;
    }

    return emptyComponent;
  }

  protected renderChildren() {
    const { isLoading, children, emptyComponent, isError } = this.props;

    if (React.Children.count(children)) {
      return children;
    }

    if (!isLoading && emptyComponent && !isError) {
      return this.renderEmptyPlaceholder();
    }

    return null;
  }

  protected renderGlobalSpinner(overlay) {
    const { isLoading, isLoadingFirstPage } = this.props;

    if (isLoading && isLoadingFirstPage) {
      return (
        <div
          className={cx(css.spinnerIntialLoadContainer, {
            [css.spinnerIntialLoadContainer_overlay]: overlay,
          })}
        >
          {InfiniteList.spinnerWithWrap}
        </div>
      );
    }

    return null;
  }

  protected renderError(style?: CSSProperties) {
    const { isLoading, isError } = this.props;

    return (
      isError && (
        <div className={css.b__error} style={style}>
          <div className={css.b__errorText}>Ошибка при загрузке. Попробуйте еще.</div>
          <Button onClick={this.handleLoad} disabled={isLoading}>
            Загрузить
          </Button>
        </div>
      )
    );
  }

  protected renderLocalSpinner(style?: CSSProperties) {
    const { isLoading, isLoadingFirstPage, isEof, isError } = this.props;

    return (
      !isEof &&
      !isError && (
        <div className={css.spinner} style={style}>
          {isLoading && !isLoadingFirstPage && InfiniteList.spinnerWithWrap}
        </div>
      )
    );
  }

  public render() {
    const {
      className,
      classNameScrollNode,
      children,
      isLoading,
      isLoadingFirstPage,
      isEof,
      isError,
      onLoad,
      onScroll,
      loadBeginOffset,
      getScrollNode,
      header,
      emptyComponent,
      inject,
      onKeyDown,
      role,
      tabIndex,
      ...other
    } = this.props;

    const spinner = this.renderLocalSpinner();

    if (inject && React.isValidElement(children)) {
      return React.cloneElement(
        React.Children.only(children),
        { getScrollNode: this.getScrollNode },
        spinner,
      );
    }

    const content = this.renderChildren();
    const globalSpinner = this.renderGlobalSpinner(Boolean(content));

    const isInitLoad = !content && globalSpinner;

    return (
      <div className={cx(css.b, className, { [css.b_initLoad]: isInitLoad })} {...other}>
        <div
          className={cx(css.b__content, classNameScrollNode)}
          onKeyDown={onKeyDown}
          role={role}
          tabIndex={tabIndex}
          ref={this.getScrollNode}
        >
          {this.props.header}
          {content}
          {this.renderError()}
          {spinner}
        </div>
        {globalSpinner}
      </div>
    );
  }
}

export default InfiniteList;
