/* eslint-disable react/no-find-dom-node */
import React, { Children, cloneElement, KeyboardEventHandler, FocusEventHandler } from 'react';
import ReactDOM from 'react-dom';
import { Cancelable } from 'lodash';
import debounce from 'lodash/debounce';
import cx from 'classnames';
import getDisplayName from 'utils/getDisplayName';
import css from './withListController.modules.scss';
import { GetScrollNode, ScrollNodeType as _ScrollNodeType } from '../types';

type Id = number | string;

export type ScrollNodeType = _ScrollNodeType;

export interface WrappedComponentProps<T = ScrollNodeType> {
  onKeyDown: KeyboardEventHandler<T>;
  onFocus: FocusEventHandler<T>;
  onBlur: FocusEventHandler<T>;
  role: string;
  tabIndex: number;
  className: string;
  classNameScrollNode: string;
  getScrollNode: GetScrollNode;
}

type HandleEnter = (selected?: Id) => void;

// eslint-disable-next-line @typescript-eslint/no-explicit-any
type HandleChange = (id: Id, childProps: any) => void;
type HandleChangeCancelable = HandleChange & Cancelable;

interface ListControllerProps {
  selected?: Id;
  children?: React.ReactNode;
  onChange?: HandleChange;
  onEnter?: HandleEnter;
  onKeyDown?: (event: KeyboardEventHandler<ScrollNodeType>, selected?: Id) => void;
  onEscape?: () => void;
  getScrollNode?: GetScrollNode;
  onChangeDelay?: number;
  className?: string;
  classNameScrollNode?: string;
  autoSelectFirst?: boolean;
  focusOnSelectFirst?: boolean;
}

interface ListControllerState {
  selected?: Id;
}

function isCancelableHandleChange(
  handle: HandleChange | HandleChangeCancelable,
): handle is HandleChangeCancelable {
  return Boolean((handle as HandleChangeCancelable).cancel);
}

function withListController<P extends {}>(config?: Partial<ListControllerProps>) {
  return (WrappedComponent: React.ComponentType<P & Partial<WrappedComponentProps>>) => {
    class ListController extends React.Component<ListControllerProps, ListControllerState> {
      public static displayName = `withListController(${getDisplayName(WrappedComponent)})`;

      public static defaultProps = {
        selected: null,
        onChangeDelay: 200,
        autoSelectFirst: false,
        focusOnSelectFirst: false,
        ...config,
      };

      private static isEqual = (selected1, selected2) => {
        const fixSelected1 = selected1 == null ? null : selected1;
        const fixSelected2 = selected2 == null ? null : selected2;

        return fixSelected1 === fixSelected2;
      };

      private static getFirstId = (props?: ListControllerProps) => {
        if (!props) {
          return null;
        }

        const { children } = props;

        if (Children.count(children) === 0) {
          return null;
        }

        const firstChild = Children.toArray(children)[0];

        if (!React.isValidElement(firstChild)) {
          return null;
        }

        const { id } = firstChild.props;

        return id != null ? id : null;
      };

      private wrapNode: ScrollNodeType;

      private childrenKeys: Id[] = [];

      private handleChange: HandleChange | HandleChangeCancelable;

      private isSelectedInChildren = false;
      // eslint-disable-next-line @typescript-eslint/no-explicit-any
      private selectedNode: any;
      // eslint-disable-next-line @typescript-eslint/no-explicit-any
      private childrenProps: any;

      public constructor(props) {
        super(props);

        const handleChange: HandleChange = (id, p) => {
          if (typeof this.props.onChange === 'function') {
            this.props.onChange(id, p);
          }
        };

        if (props.onChangeDelay) {
          this.handleChange = debounce(handleChange, props.onChangeDelay);
        } else {
          this.handleChange = handleChange;
        }

        this.state = {
          selected: props.selected,
        };
      }

      public componentDidMount() {
        this.selectFirstIfNeeded();
      }

      public componentWillReceiveProps(nextProps) {
        const nextSelected = nextProps.selected == null ? null : nextProps.selected;

        if (nextSelected !== this.state.selected && nextSelected !== this.props.selected) {
          this.setState({ selected: nextSelected });
        }
      }

      public componentDidUpdate(prevProps) {
        this.selectFirstIfNeeded(prevProps);
      }

      public componentWillUnmount() {
        if (isCancelableHandleChange(this.handleChange)) {
          this.handleChange.cancel();
        }
      }

      private getCurrentIndex() {
        if (this.childrenKeys.length && this.state.selected) {
          return this.childrenKeys.indexOf(this.state.selected);
        }

        return -1;
      }

      private getScrollNode: GetScrollNode = (node) => {
        if (typeof this.props.getScrollNode === 'function') {
          this.props.getScrollNode(node);
        }
        this.wrapNode = node;
      };

      private setSelectedRef = (node) => {
        this.selectedNode = node;
      };

      private scrollToSelected = () => {
        if (this.selectedNode) {
          const selectedNode = ReactDOM.findDOMNode(this.selectedNode) as HTMLElement | null;
          if (selectedNode) {
            const cO = selectedNode.offsetTop;
            const cH = selectedNode.offsetHeight;
            const pH = this.wrapNode.clientHeight;
            const pS = this.wrapNode.scrollTop;
            if (cO + cH > pH + pS) {
              this.wrapNode.scrollTop = cO - pH + cH;
            } else if (cO < pS) {
              this.wrapNode.scrollTop = cO;
            }
          }
        }
      };

      private handleKeyDown = (e) => {
        if (e.key === 'ArrowUp') {
          this.move(-1);
          e.preventDefault();
        } else if (e.key === 'ArrowDown') {
          this.move(1);
          e.preventDefault();
        } else if (
          e.key === 'Enter' &&
          typeof this.props.onEnter === 'function' &&
          this.state.selected != null
        ) {
          this.props.onEnter(this.state.selected);
        } else if (e.key === 'Escape' && typeof this.props.onEscape === 'function') {
          this.props.onEscape();
        }

        const { onKeyDown } = this.props;
        if (this.state.selected != null && typeof onKeyDown === 'function') {
          onKeyDown(e, this.state.selected);
        }
      };

      private move(offset) {
        if (offset !== 0) {
          const newIndex = this.getCurrentIndex() + offset;
          if (
            (offset > 0 && newIndex < this.childrenKeys.length) ||
            (offset < 0 && newIndex >= 0)
          ) {
            this.changeId(this.childrenKeys[newIndex]);
          }
        }
      }

      private changeId(newId) {
        if (!ListController.isEqual(newId, this.state.selected)) {
          this.setState({ selected: newId }, this.scrollToSelected);
        }
        if (!ListController.isEqual(newId, this.props.selected)) {
          this.handleChange(newId, newId === null ? null : this.childrenProps[newId]);
        }
      }

      private selectFirstIfNeeded(prevProps?: ListControllerProps) {
        const { props } = this;

        if (props.autoSelectFirst && (!this.isSelectedInChildren || this.state.selected == null)) {
          const prevFirstId = ListController.getFirstId(prevProps);
          const firstId = ListController.getFirstId(props);

          if (firstId !== prevFirstId) {
            if (firstId != null) {
              if (props.focusOnSelectFirst) {
                this.wrapNode.focus();
              }
              this.changeId(firstId);
            } else {
              this.changeId(null);
            }
          }
        }
      }

      public render() {
        const { children, className, classNameScrollNode, ...passThroughProps } = this.props;
        delete passThroughProps.selected;
        delete passThroughProps.onChange;
        delete passThroughProps.onEnter;
        delete passThroughProps.onKeyDown;
        delete passThroughProps.onEscape;
        delete passThroughProps.getScrollNode;
        delete passThroughProps.onChangeDelay;
        delete passThroughProps.autoSelectFirst;
        delete passThroughProps.focusOnSelectFirst;

        let newChildren;

        this.isSelectedInChildren = false;
        if (Children.count(children)) {
          this.childrenKeys = [];
          this.childrenProps = {};
          newChildren = Children.map(children, (child) => {
            if (!React.isValidElement(child)) {
              return child;
            }

            if (child.props.id != null) {
              this.childrenKeys.push(child.props.id);
              this.childrenProps[child.props.id] = child.props;
            }

            const isSelected = this.state.selected === child.props.id;
            this.isSelectedInChildren = this.isSelectedInChildren || isSelected;
            return cloneElement(child, {
              selected: isSelected,
              ref: this.state.selected === child.props.id ? this.setSelectedRef : undefined,
            });
          });
        } else {
          newChildren = this.props.children;
          this.childrenProps = null;
          this.childrenKeys = [];
        }

        const props = {
          onKeyDown: this.handleKeyDown,
          role: 'presentation',
          tabIndex: 0,
          className: cx(css.b, className),
          classNameScrollNode: cx(css.b__content_interactive, classNameScrollNode),
          ...passThroughProps,
          getScrollNode: this.getScrollNode,
        };

        return <WrappedComponent {...((props as unknown) as P)}>{newChildren}</WrappedComponent>;
      }
    }

    return ListController;
  };
}

export default withListController;
