import React, { useCallback, useState, useRef } from 'react';
import { storiesOf } from '@storybook/react';
import InfiniteListVirtual from './InfiniteVirtualList';

interface ProviderConfig {
  pageLength: number;
  totalCount: number;
}

interface Item {
  id: number;
  text: string;
}

interface LoadData {
  eof: boolean;
  items: Item[];
}

class Provider {
  private static defaultConfig: ProviderConfig = {
    pageLength: 50,
    totalCount: 1000,
  };

  private config: ProviderConfig;

  private items: Item[];

  public constructor(props?: Partial<ProviderConfig>) {
    this.config = { ...Provider.defaultConfig, ...props };

    this.items = Array(this.config.totalCount)
      .fill(true)
      .map((_, index) => ({ id: index, text: `Item ${index}` }));
  }

  public load(offset: number = 0): LoadData {
    const lastIndex = offset + this.config.pageLength;

    return { items: this.items.slice(offset, lastIndex), eof: lastIndex >= this.items.length };
  }

  public asyncLoad(offset: number = 0): Promise<LoadData> {
    console.log('async load. offset = ', offset);

    return new Promise(resolve => {
      setTimeout(() => {
        resolve(this.load(offset));
      }, 100);
    });
  }
}

const provider = new Provider();

const Row = ({ index, style, data }) => <div style={style}>{data[index].text}</div>;

const style = { height: 600, width: 300 };

storiesOf('InfiniteList', module).add('InfiniteListVirtual', () => {
  const [items, setItems] = useState<Item[]>(provider.load().items);
  const [eof, setEof] = useState(false);
  const [size, setSize] = useState(100);
  const [isLoading, setLoading] = useState(false);
  const list = useRef<InfiniteListVirtual>(null);

  const handleLoad = useCallback(() => {
    setLoading(true);

    provider.asyncLoad(items.length).then(data => {
      setLoading(false);

      setEof(data.eof);
      setItems(items.concat(data.items));

      return data;
    });
  }, [items]);

  const handleSetSize = useCallback(() => {
    const input = document.getElementById('size');

    if (input instanceof HTMLInputElement) {
      setSize(Number(input.value));
      if (list.current) {
        list.current.resetAfterIndex(0);
      }
    }
  }, []);

  const handleRemoveItem = useCallback(() => {
    const input = document.getElementById('index');

    if (input instanceof HTMLInputElement) {
      const index = Number(input.value);

      setItems([...items.slice(0, index), ...items.slice(index + 1)]);
    }
  }, [items]);

  const getItemSize = useCallback(
    (index: number) => {
      console.log(`getItemSize: ${index}`);
      return size;
    },
    [size],
  );

  const getItemKey = useCallback(
    (index, data) => {
      const item = data[index];

      return item.id;
    },
    [items],
  );

  return (
    <div style={{ display: 'flex', flexFlow: 'row nowrap' }}>
      <div>
        <InfiniteListVirtual
          isEof={eof}
          onLoad={handleLoad}
          itemCount={items.length}
          isLoading={isLoading}
          style={style}
          itemData={items}
          itemKey={getItemKey}
          itemSize={getItemSize}
          ref={list}
          estimatedItemSize={200}
        >
          {Row}
        </InfiniteListVirtual>
      </div>
      <div style={{ marginLeft: 50 }}>
        <h1>Tools</h1>
        <h2>Remove item</h2>
        <input type="text" id="index" placeholder="index" />
        <button type="button" onClick={handleRemoveItem}>
          del
        </button>
        <h2>Change item size</h2>
        <input type="text" id="size" placeholder="size" />
        <button type="button" onClick={handleSetSize}>
          set size
        </button>
      </div>
    </div>
  );
});
