import React, { ComponentType, CSSProperties } from 'react';
import AutoSizer from 'react-virtualized-auto-sizer';
import {
  VariableSizeList as List,
  ListChildComponentProps,
  ListItemKeySelector,
} from 'react-window';
import cx from 'classnames';
import css from 'components/InfiniteList/Original/InfiniteList.module.css';
import { WrappedComponentProps as KeyboardWrappedComponentProps } from './withKeyboardNavigation';

import {
  renderError,
  renderLocalSpinner,
  renderGlobalSpinner,
  hasLocalSpinner,
  hasError,
} from '../renderComponents';

export interface InfiniteListVirtualProps {
  className?: string;
  classNameScrollNode?: string;
  isEof?: boolean;
  isError?: boolean;
  isLoading?: boolean;
  isLoadingFirstPage?: boolean;
  header?: React.ReactNode;
  children?: React.ReactNode;
  onLoad: () => void;
  emptyComponent?: React.ReactNode | React.ComponentType<{}>;
  itemCount: number;
  // eslint-disable-next-line
  itemData?: any;
  itemKey?: ListItemKeySelector;
  itemSize: (index: number) => number;
  estimatedItemSize?: number;
  style?: CSSProperties;
}

type Props = InfiniteListVirtualProps & KeyboardWrappedComponentProps;

export class InfiniteListVirtual extends React.Component<Props> {
  private static ERROR_KEY = 'error';

  private static SPINNER_KEY = 'spinner';

  public static defaultProps = {
    itemSize: () => 100,
  };

  public list = React.createRef<List>();

  private isFocused: boolean;

  private itemCount: number = 0;

  private loadingIndexThreshold: number = 2;

  private errorIndex: number = -1;

  private localSpinnerIndex: number = -1;

  public componentDidMount() {
    const { onKeyDown } = this.props;
    if (onKeyDown) {
      document.addEventListener('keydown', this.handleKeyDown);
    }
  }

  public componentDidUpdate(prevProps) {
    if (
      this.props.isLoadingFirstPage === true &&
      prevProps.isLoadingFirstPage !== this.props.isLoadingFirstPage
    ) {
      this.scrollTo(0);
    }
  }

  public componentWillUnmount() {
    if (this.props.onKeyDown) {
      document.removeEventListener('keydown', this.handleKeyDown);
    }
  }

  private handleKeyDown = (event) => {
    const { onKeyDown } = this.props;
    if (this.isFocused && onKeyDown) {
      onKeyDown(event);
    }
  };

  private getItemKey: ListItemKeySelector = (index, data) => {
    if (index === this.errorIndex) {
      return InfiniteListVirtual.ERROR_KEY;
    }

    if (index === this.localSpinnerIndex) {
      return InfiniteListVirtual.SPINNER_KEY;
    }

    if (this.props.itemKey) {
      return this.props.itemKey(index, data);
    }

    return -1;
  };

  private getItemCount = () => {
    const { isEof, isError } = this.props;

    let count = this.props.itemCount;

    if (hasLocalSpinner({ isEof, isError })) {
      this.localSpinnerIndex = count;
      count += 1;
    } else {
      this.localSpinnerIndex = -1;
    }

    if (hasError({ isError })) {
      this.errorIndex = count;
      count += 1;
    } else {
      this.errorIndex = -1;
    }

    this.itemCount = count;

    return count;
  };

  private getItemSize: (index: number) => number = (index) => {
    if (index === this.errorIndex || index === this.localSpinnerIndex) {
      return 100;
    }

    return this.props.itemSize(index);
  };

  private handleItemsRendered = ({ visibleStopIndex }) => {
    if (
      !this.props.isLoading &&
      this.localSpinnerIndex !== -1 &&
      visibleStopIndex >= this.localSpinnerIndex - this.loadingIndexThreshold
    ) {
      this.props.onLoad();
    }
  };

  private handleFocus = () => {
    this.isFocused = true;
  };

  private handleBlur = () => {
    this.isFocused = false;
  };

  public resetAfterIndex(index: number, shouldForceUpdate: boolean = true): void {
    if (this.list.current) {
      this.list.current.resetAfterIndex(index, shouldForceUpdate);
    }
  }

  public scrollTo(scrollOffset: number): void {
    if (this.list.current) {
      this.list.current.scrollTo(scrollOffset);
    }
  }

  private renderRow: React.FC<ListChildComponentProps> = (props) => {
    const { index, style } = props;
    const { isLoading, isError, isEof, isLoadingFirstPage, onLoad } = this.props;

    if (index === this.errorIndex) {
      return renderError({ style, isError, isLoading, onLoad });
    }

    if (index === this.localSpinnerIndex) {
      return renderLocalSpinner({ style, isEof, isError, isLoading, isLoadingFirstPage });
    }

    return React.createElement(
      this.props.children as ComponentType<ListChildComponentProps>,
      props,
    );
  };

  public render() {
    const {
      className,
      classNameScrollNode,
      isLoading,
      isLoadingFirstPage,
      itemData,
      itemKey,
      style,
      estimatedItemSize,
    } = this.props;

    const globalSpinner = renderGlobalSpinner({
      isOverlay: true,
      isLoading,
      isLoadingFirstPage,
    });

    const isInitLoad = Boolean(globalSpinner);

    return (
      <div
        className={cx(css.b, className, { [css.b_initLoad]: isInitLoad })}
        style={style}
        onFocus={this.handleFocus}
        onBlur={this.handleBlur}
        data-testid="InfiniteVirtualList"
      >
        <AutoSizer>
          {({ height, width }) => (
            <List
              ref={this.list}
              height={height}
              width={width}
              itemCount={this.getItemCount()}
              itemSize={this.getItemSize}
              estimatedItemSize={estimatedItemSize}
              itemData={itemData}
              itemKey={itemKey ? this.getItemKey : undefined}
              className={classNameScrollNode}
              onItemsRendered={this.handleItemsRendered}
            >
              {this.renderRow}
            </List>
          )}
        </AutoSizer>
        {globalSpinner}
      </div>
    );
  }
}

export default InfiniteListVirtual;
