import * as React from 'react';
import { VariableSizeList } from 'react-window';
import VirtualList from './InfiniteVirtualList';

type Id = string | number;

interface Props {
  listRef: React.RefObject<VirtualList>;
  itemIds: Id[];
  selectedId?: Id;
  onChange?: (id: Id) => void;
}

interface State {
  selectedId: Id;
}

export interface WrappedComponentProps {
  itemData?: boolean[];
  onKeyDown?: (event: KeyboardEvent) => void | React.KeyboardEventHandler;
}

const isId = (anything: any): anything is Id =>
  typeof anything === 'string' || typeof anything === 'number';

const withKeyboardNavigation = <P extends WrappedComponentProps>(
  WrappedComponent: React.ComponentType<P>,
): React.ComponentType<P & Props> => {
  class KeyboardNavigation extends React.Component<P & Props, State> {
    // InfiniteVirtualList
    private virtualList: React.RefObject<{
      list: React.RefObject<VariableSizeList>;
    }>;

    public constructor(props: P & Props) {
      super(props);
      this.virtualList = props.listRef;
      this.state = {
        selectedId: props.selectedId == null ? -1 : props.selectedId,
      };
    }

    public componentWillReceiveProps(nextProps: P & Props) {
      const nextSelected = nextProps.selectedId;

      if (
        nextSelected !== this.state.selectedId &&
        nextSelected !== this.props.selectedId &&
        isId(nextSelected)
      ) {
        this.setState({ selectedId: nextSelected });
      }
    }

    private getCurrentIndex = (): number => this.props.itemIds.indexOf(this.state.selectedId);

    private scrollToSelectedItem = (): void => {
      const virtualList = this.virtualList && this.virtualList.current;
      const innerList = virtualList && virtualList.list.current;
      if (innerList) {
        innerList.scrollToItem(this.getCurrentIndex(), 'smart');
      }
    };

    private changeId = (newId: Id): void => {
      if (newId !== this.state.selectedId) {
        this.setState({ selectedId: newId }, this.scrollToSelectedItem);

        const { onChange } = this.props;
        if (onChange) {
          onChange(newId);
        }
      }
    };

    private move = (offset: number): void => {
      if (offset === 0) {
        return;
      }

      const { itemIds } = this.props;
      const newIndex = this.getCurrentIndex() + offset;
      if ((offset > 0 && newIndex < itemIds.length) || (offset < 0 && newIndex >= 0)) {
        this.changeId(itemIds[newIndex]);
      }
    };

    private handleKeyDown = (event: React.KeyboardEvent | KeyboardEvent): void => {
      if (event.key === 'ArrowUp') {
        this.move(-1);
        event.preventDefault();
      }

      if (event.key === 'ArrowDown') {
        this.move(1);
        event.preventDefault();
      }
    };

    public render(): React.ReactNode {
      const { listRef } = this.props;
      const selectedItemData = this.props.itemIds.map(itemId => itemId === this.state.selectedId);
      return (
        <WrappedComponent
          {...this.props}
          ref={listRef}
          itemData={selectedItemData}
          onKeyDown={this.handleKeyDown}
        />
      );
    }
  }

  return KeyboardNavigation;
};

export default withKeyboardNavigation;
