import React, { CSSProperties } from 'react';
import Button from '@crm/components/dist/lego2/Button';
import cx from 'classnames';
import Spinner from 'components/Spinner';
import css from 'components/InfiniteList/Original/InfiniteList.module.css';

export interface RenderEmptyPlaceholder {
  emptyComponent?: React.ReactNode | React.ComponentType<{}>;
}

type RenderFunc<P, T> = (props: P) => T;

export const renderEmptyPlaceholder: RenderFunc<RenderEmptyPlaceholder, React.ReactNode> = ({
  emptyComponent,
}) => {
  if (typeof emptyComponent === 'function') {
    const EmptyComponent = emptyComponent as React.ComponentType<{}>;
    return <EmptyComponent />;
  }

  return emptyComponent;
};

interface RenderChildren {
  isLoading?: boolean;
  children?: React.ReactNode;
  isError?: boolean;
}

export const renderChildren: RenderFunc<
  RenderChildren & RenderEmptyPlaceholder,
  React.ReactNode
> = ({ isLoading, children, emptyComponent, isError }) => {
  if (React.Children.count(children)) {
    return children;
  }

  if (!isLoading && emptyComponent && !isError) {
    return renderEmptyPlaceholder({ emptyComponent });
  }

  return null;
};

const spinnerWithWrap = (
  <div className={css.spinnerWrap}>
    <Spinner visible modal={false} />
  </div>
);

interface RenderGlobalSpinner {
  isLoading?: boolean;
  isOverlay?: boolean;
  isLoadingFirstPage?: boolean;
}

export const renderGlobalSpinner: RenderFunc<RenderGlobalSpinner, React.ReactNode> = ({
  isLoading,
  isLoadingFirstPage,
  isOverlay,
}) => {
  if (isLoading && isLoadingFirstPage) {
    return (
      <div
        className={cx(css.spinnerIntialLoadContainer, {
          [css.spinnerIntialLoadContainer_overlay]: isOverlay,
        })}
      >
        {spinnerWithWrap}
      </div>
    );
  }

  return null;
};

export const hasError: (props: { isError?: boolean }) => boolean = ({ isError }) =>
  Boolean(isError);

interface RenderError {
  isLoading?: boolean;
  isError?: boolean;
  style?: CSSProperties;
  onLoad: () => void;
}

export const renderError: RenderFunc<RenderError, React.ReactElement | null> = ({
  isLoading,
  isError,
  style,
  onLoad,
}) => {
  return hasError({ isError }) ? (
    <div className={css.b__error} style={style}>
      <div className={css.b__errorText}>Ошибка при загрузке. Попробуйте еще.</div>
      <Button onClick={onLoad} disabled={isLoading}>
        Загрузить
      </Button>
    </div>
  ) : null;
};

export const hasLocalSpinner: (props: { isEof?: boolean; isError?: boolean }) => boolean = ({
  isEof,
  isError,
}) => !isEof && !isError;

interface RenderLocalSpinner {
  isLoading?: boolean;
  isLoadingFirstPage?: boolean;
  isEof?: boolean;
  isError?: boolean;
  style?: CSSProperties;
}

export const renderLocalSpinner: RenderFunc<RenderLocalSpinner, React.ReactElement | null> = ({
  isLoading,
  isLoadingFirstPage,
  isEof,
  isError,
  style,
}) => {
  return hasLocalSpinner({ isError, isEof }) ? (
    <div className={css.spinner} style={style}>
      {isLoading && !isLoadingFirstPage && spinnerWithWrap}
    </div>
  ) : null;
};
