import React, { useState, ReactNodeArray, ReactElement } from 'react';
import Button from '@crm/components/dist/lego2/Button';
import { Text } from '@crm/components/dist/lego2/Text';
import cx from 'classnames';
import Icon from '@crm/components/dist/lego2/Icon';
import { ButtonWithPopup } from 'components/Roller/RollerWithPopup/ButtonWithPopup';
import { RollerWithPopup } from 'components/Roller';
import { ExpandButton } from 'components/ExpandButton';
import { FieldItem, FieldItemWithItems } from 'modules/issues/types';
import IssueField from 'modules/issues/components/Issue/IssueField';
import CopyButton from 'lego/components/CopyButton';
import { getPin as getButtonPin } from 'lego/components/ButtonGroup/ButtonGroup';
import Access from 'utils/Access';
import createIssueUrl from 'modules/issues/components/utils/createIssueUrl';
import { PopupContext } from '@crm/components/dist/lego2/Popup/Popup';
import { config } from 'services/Config';
import getRoutingRoot from 'utils/getRoutingRoot';
import Link from '@crm/components/dist/lego2/Link';
import { Divider } from 'components/Divider';
import css from './IssueHeader.module.css';
import { IssueHeaderProps, ActionItem } from './IssueHeader.types';
import { clearListWithoutEditAccess } from './IssueHeader.utils';
import { IssueFieldPopupGroup, IssueFieldPopupGroupProps } from './IssueFieldPopupGroup';

export const IssueHeader: React.FC<IssueHeaderProps> = ({ issue }) => {
  const { data: issueData, props: issueProps } = issue;
  const [statusPopupVisible, setStatusPopupVisible] = useState(false);
  const [hiddenItemsCount, setHiddenItemsCount] = useState<number>(0);

  const quickPanelItems: FieldItem[] = clearListWithoutEditAccess(
    issueProps.quickPanel?.items,
    issueProps.quickPanel?.access,
  );
  let actionItems: ActionItem[] = clearListWithoutEditAccess(
    issueProps.actions?.items,
    issueProps.actions?.access,
  ).slice();
  const actionStateChangeItems: FieldItemWithItems[] = clearListWithoutEditAccess(
    issueProps.actionStateChange?.items,
    issueProps.actionStateChange?.access,
  );
  const starTrekItems: FieldItem[] = clearListWithoutEditAccess(
    issueProps.starTrekActions?.items,
    issueProps.starTrekActions?.access,
  );
  const canTakeIssue = Access.isEdit(issueProps.takeIssue?.access || 0);

  const downloadIndex = actionItems.findIndex((item) => item.val === 'DownloadFiles');
  if (downloadIndex !== -1) {
    const downloadItem = actionItems[downloadIndex];
    actionItems.splice(downloadIndex, 1);
    actionItems.push(downloadItem);
  }

  if (Access.isEdit(issueProps?.name?.access || 1)) {
    actionItems = actionItems.concat({
      val: 'name',
      text: 'Переименовать',
      alert: false,
      redirect: false,
      form: 'rename',
    });
  }

  const handleHiddenItemsChanged = (components: ReactElement[] | ReactNodeArray) => {
    if (components.length !== hiddenItemsCount) {
      setHiddenItemsCount(components.length);
    }
  };

  const accountName = issueData.account?.info.name;
  const accountId = issueData.account?.id;

  return (
    <div data-testid="issue-header" className={css.IssueHeader}>
      <div className={css.IssueHeader__content}>
        <div className={css.IssueHeader__info}>
          <div className={css.IssueHeader__nameRow}>
            <Text
              data-testid="IssueDataName"
              typography="subheader-s"
              weight="medium"
              color="primary"
              overflow="ellipsis"
              title={issueData.name}
            >
              {issueData.name}
            </Text>
          </div>
          <div className={css.IssueHeader__subRow}>
            <Text
              className={css.IssueHeader__idText}
              typography="body-long-m"
              weight="medium"
              color="secondary"
              overflow="ellipsis"
            >
              {issueData.id}
            </Text>
            <CopyButton
              className={css.IssueHeader__idCopyButton}
              copy={createIssueUrl({
                issueId: issueData.id,
                issueTypeId: issueData.typeId,
              })}
              iconProvider={(cls) => <Icon className={cls} svg="copy2" svgSize="ms" />}
              title="Скопировать ссылку"
              tooltipText="Ссылка скопирована"
              withTooltip
              tooltipCloseDelay={1500}
              size="xs"
              view="clear"
              stopPropagation
            />

            <Text
              className={css.IssueHeader__stateName}
              typography="body-long-m"
              weight="medium"
              color="secondary"
              overflow="ellipsis"
            >
              {issueData.state.name}
            </Text>

            {accountId && (
              <Link
                className={css.IssueHeader__accountName}
                href={`${getRoutingRoot()}/account/${accountId}`}
              >
                <Text
                  typography="body-long-m"
                  weight="medium"
                  color="secondary"
                  overflow="ellipsis"
                  data-testid="AccountName"
                  className={css.IssueHeader__accountNameText}
                >
                  {accountName}
                </Text>
              </Link>
            )}
          </div>
        </div>
        <div className={css.IssueHeader__actions}>
          <RollerWithPopup<IssueFieldPopupGroupProps>
            className={css.IssueHeader__rollerWithPopup}
            rollerClassNames={{
              containerClassName: css.IssueHeader__rollerContainer,
            }}
            moreButtonProps={{
              element: (
                <ExpandButton
                  expanded={statusPopupVisible}
                  className={css.IssueHeader__action}
                  view="default"
                  size="m"
                  text={
                    <Text typography="body-long-m" weight="medium" color="primary">
                      Другие статусы
                    </Text>
                  }
                  pin={getButtonPin(
                    actionStateChangeItems.length - hiddenItemsCount + 1,
                    actionStateChangeItems.length - hiddenItemsCount,
                  )}
                />
              ),
            }}
            popupProps={{
              className: css.IssueHeader__popup,
              containerClassName: css.IssueHeader__popupContainer,
              element: (props) => (
                <PopupContext.Consumer>
                  {({ onClose }) => (
                    <IssueFieldPopupGroup
                      {...props}
                      isInRoller
                      onClick={onClose as () => void}
                      groupMode="list"
                      buttonClassName={cx(css.IssueHeader__popupAction, css.IssueHeader__popupItem)}
                      listClassNames={{
                        child: css.IssueHeader__popupAction,
                        container: css.IssueHeader__popupItem,
                        title: css.IssueHeader__popupTitle,
                      }}
                    />
                  )}
                </PopupContext.Consumer>
              ),
              directions: ['bottom-center'],
              onVisibleChange: setStatusPopupVisible,
            }}
            updateKey={actionStateChangeItems}
            onHiddenComponentsChange={handleHiddenItemsChanged}
          >
            {actionStateChangeItems.map((actionStateChangeItem, index) => (
              <IssueFieldPopupGroup
                key={actionStateChangeItem.val}
                name="actionStateChange"
                fieldItem={actionStateChangeItem}
                size="m"
                pin={getButtonPin(actionStateChangeItems.length, index)}
                groupMode="popup"
                buttonClassName={css.IssueHeader__action}
                popupClassNames={{
                  button: css.IssueHeader__action,
                  child: css.IssueHeader__popupAction,
                  container: css.IssueHeader__popupContainer,
                  popup: css.IssueHeader__popup,
                }}
              />
            ))}
          </RollerWithPopup>

          {canTakeIssue && (
            <IssueField
              name="takeIssue"
              component="Button2"
              className={cx(css.IssueHeader__action, css.IssueHeader__actionIcon)}
              fieldProps={{
                size: 'm',
                view: 'default',
              }}
              // @ts-ignore
              url={`/actions/issue/${issueData.id}/take`}
              redirect={false}
            >
              Взять себе
            </IssueField>
          )}

          <IssueField
            name="supportChat"
            component={
              config.value.features.enableInternalChats ? 'SupportChatIssue' : 'SupportChat'
            }
            className={cx(css.IssueHeader__action, css.IssueHeader__actionIcon)}
            fieldProps={{
              size: 'm',
              view: 'default',
            }}
          />

          {starTrekItems && starTrekItems.length > 0 && (
            <ButtonWithPopup
              button={
                <Button
                  data-testid="starTrekButton"
                  className={cx(css.IssueHeader__action, css.IssueHeader__actionIcon)}
                  size="m"
                  view="default"
                  icon={() => <Icon svgSize="m" svg="starTrek" />}
                />
              }
              popupProps={{
                directions: ['bottom-center', 'bottom-right'],
                className: css.IssueHeader__popup,
              }}
            >
              <PopupContext.Consumer>
                {({ onClose }) => (
                  <div className={css.IssueHeader__popupContainer}>
                    {starTrekItems?.map((starTrekItem) => (
                      <IssueField
                        key={starTrekItem.val}
                        name="starTrekActions"
                        backendUpdateKey={starTrekItem.val}
                        component="Button2"
                        fieldProps={{
                          view: 'clear',
                          asButton: true,
                          onClick: onClose,
                        }}
                        className={cx(css.IssueHeader__popupAction)}
                        {...starTrekItem}
                      >
                        {starTrekItem.text}
                      </IssueField>
                    ))}
                  </div>
                )}
              </PopupContext.Consumer>
            </ButtonWithPopup>
          )}

          <ButtonWithPopup
            button={
              <Button
                data-testid="MoreButton"
                className={cx(css.IssueHeader__action, css.IssueHeader__actionIcon)}
                size="m"
                view="default"
                icon={() => <Icon svgSize="m" svg="ellipsis" />}
              />
            }
            popupProps={{
              directions: ['bottom-center', 'bottom-right'],
              className: css.IssueHeader__popup,
            }}
          >
            <PopupContext.Consumer>
              {({ onClose }) => (
                <div className={css.IssueHeader__popupContainer}>
                  {quickPanelItems?.map((quickPanelItem) => (
                    <IssueField
                      key={quickPanelItem.val}
                      name="quickPanel"
                      backendUpdateKey={quickPanelItem.val}
                      component="Button2"
                      fieldProps={{
                        view: 'clear',
                        asButton: true,
                        onClick: onClose,
                      }}
                      className={cx(css.IssueHeader__popupAction)}
                      {...quickPanelItem}
                    >
                      {quickPanelItem.text}
                    </IssueField>
                  ))}
                  {actionItems?.map((actionItem) => (
                    <>
                      {actionItem.val === 'linkIssue' && <Divider />}
                      {actionItem.val === 'DownloadFiles' && <Divider />}
                      {actionItem.val === 'ActionCreatePlannedMeeting' && <Divider />}

                      <IssueField
                        key={actionItem.val}
                        name="actions"
                        backendUpdateKey={actionItem.val}
                        component="Button2"
                        fieldProps={{
                          view: 'clear',
                          asButton: true,
                          onClick: onClose,
                        }}
                        className={cx(css.IssueHeader__popupAction)}
                        {...actionItem}
                      >
                        {actionItem.text}
                      </IssueField>
                    </>
                  ))}
                </div>
              )}
            </PopupContext.Consumer>
          </ButtonWithPopup>
        </div>
      </div>
    </div>
  );
};
