import React from 'react';
import PropTypes from 'prop-types';
import ReactDOM from 'react-dom';
import cx from 'classnames';
import FilesList from 'components/FilesList';
import CalendarEvent from 'components/CalendarEvent';
import defaultHeader from './Header';
import Info from './Info';
import QuoteWrapper from './QuoteWrapper';
import linkFormatter from './utils/linkFormatter';
import quotesFormatter from './utils/quotesFormatter';
import cssDefault from './styles.modules.scss';
import cssMailPage from './mailPage.modules.scss';

const propTypes = {
  theme: PropTypes.string,
  showActions: PropTypes.bool,
  header: PropTypes.func,
  objectType: PropTypes.string,
  calendarTip: CalendarEvent.propTypes.event,
};

const defaultProps = {
  theme: 'DEFAULT',
  showActions: false,
  header: defaultHeader,
  body: '',
  objectType: 'mail',
};

const CSS_THEME = {
  DEFAULT: cssDefault,
  MAIL_MODULE: cssMailPage,
};

export default class Mail extends React.Component {
  constructor(props) {
    super(props);

    if (props.isHtml) {
      const formattedBody = linkFormatter(props.body);
      const mailId = props.id;
      const { html: quoteFormattedBody, ids } = quotesFormatter(formattedBody, mailId);
      this.quoteIds = ids;

      this.body = quoteFormattedBody;
    }
  }

  componentDidMount() {
    if (Array.isArray(this.quoteIds)) {
      this.quoteIds.forEach(quoteInfo => {
        const quoteEl = this.nodeBody.querySelector(`#${quoteInfo.id}`);
        ReactDOM.render(React.createElement(QuoteWrapper, quoteInfo.props), quoteEl);
      });
    }
  }

  componentWillUnmount() {
    if (Array.isArray(this.quoteIds)) {
      this.quoteIds.forEach(quoteInfo => {
        const quoteEl = this.nodeBody.querySelector(`#${quoteInfo.id}`);
        ReactDOM.unmountComponentAtNode(quoteEl);
      });
    }
  }

  render() {
    const {
      className,
      from,
      type,
      midX,
      to,
      subject,
      cc,
      bcc,
      files,
      tags,
      id,
      date,
      body,
      isHtml,
      header: Header,
      showMeta,
      extraMeta,
      delayedDate,
      isImportant,
      fromDecoratorFunc,
      objectType,
      calendarTip,
    } = this.props;

    const css = CSS_THEME[this.props.theme];

    const renderInfo = props => (
      <Info
        id={id}
        date={date}
        type={type}
        midX={midX}
        className={props.className}
        theme={css}
        from={from}
        fromDecoratorFunc={fromDecoratorFunc}
        to={to}
        cc={cc}
        bcc={bcc}
        subject={subject}
        showMeta={showMeta}
        extraMeta={extraMeta}
        delayedDate={delayedDate}
      />
    );

    const renderFiles = props => (
      <FilesList
        files={files}
        isInline
        objectId={id}
        objectType={objectType}
        maxHeight={100}
        className={cx(css.files, props.className)}
      />
    );

    return (
      <div className={cx(css.root, className)}>
        <Header
          className={css.header}
          id={id}
          info={renderInfo}
          files={renderFiles}
          tags={tags}
          isImportant={isImportant}
          props={this.props}
        />
        <div className={css.body}>
          <CalendarEvent event={calendarTip} />
          {isHtml ? (
            <div
              ref={node => {
                this.nodeBody = node;
              }}
              dangerouslySetInnerHTML={{ __html: this.body }}
            />
          ) : (
            <pre>{body}</pre>
          )}
        </div>
      </div>
    );
  }
}

Mail.defaultProps = defaultProps;
Mail.propTypes = propTypes;
