import React from 'react';
import PropTypes from 'prop-types';
import { injectIntl } from 'react-intl';
import Button from '@crm/components/dist/lego2/Button';
import Icon from '@crm/components/dist/lego2/Icon';
import StaffCard from 'components/StaffCard';
import infoCss from 'styles/component/info.scss';

class ManagersRow extends React.Component {
  render() {
    const { managers, children, showPhoneStatus } = this.props;

    const label = this.props.intl.formatMessage(
      { id: 'ID_MANAGER' },
      { managersCount: managers.length },
    );
    let managersList = managers.map((item, index) => {
      const arr = [];
      let element = null;

      const manager = (
        <React.Fragment key={item.id}>
          <StaffCard user={item} crmPosition showPhoneStatus={showPhoneStatus} />
        </React.Fragment>
      );

      if (item.binded) {
        element = (
          <span className="binded_user" key={item.id}>
            {manager}&nbsp;
            {item.allowedToUnbind && this.props.onToggleBinding && (
              <Button
                view="pseudo"
                onClick={() => this.props.onToggleBinding(item.id, false)}
                size="xs"
                icon={(iconCls) => <Icon className={iconCls} type="cross" />}
              />
            )}
          </span>
        );
      } else {
        element = manager;
      }
      arr.push(element);
      if (index !== managers.length - 1) {
        arr.push(', ');
      }
      return arr;
    });

    if (managers.length) {
      managersList = <React.Fragment>{managersList}</React.Fragment>;
    }

    if (this.props.showBinding) {
      managersList = (
        <span>
          <Button size="xs" onClick={() => this.props.onToggleBinding(null, true)}>
            +
          </Button>
          {managers.length > 0 && <span>&nbsp;</span>}
          {managersList}
        </span>
      );
    } else if (managers.length === 0) {
      managersList = <span>—</span>;
    }

    if (typeof children === 'function') {
      return children(label, managersList);
    }

    return (
      <div className={this.props.className}>
        <span className={this.props.labelClassName}>{label}</span>
        <div className={this.props.managersClassName}>{managersList}</div>
      </div>
    );
  }
}

/**
 * @type {object}
 * @prop {string} className - root className
 * @prop {string} labelClassName - label className
 * @prop {string} managersClassName - managers className
 * @prop {RenderView} children - render function
 */

ManagersRow.propTypes = {
  className: PropTypes.string,
  labelClassName: PropTypes.string,
  managersClassName: PropTypes.string,
  children: PropTypes.func,
  managers: PropTypes.arrayOf(
    PropTypes.shape({
      id: PropTypes.number,
      name: PropTypes.name,
      login: PropTypes.login,
    }),
  ),
  intl: PropTypes.instanceOf(Object).isRequired,
  onToggleBinding: PropTypes.func,
  showBinding: PropTypes.bool,
  showPhoneStatus: PropTypes.bool,
};
ManagersRow.defaultProps = {
  className: infoCss.group,
  labelClassName: infoCss.label,
  managersClassName: infoCss.field,
  children: undefined,
  managers: [],
  onToggleBinding: undefined,
  showBinding: false,
  showPhoneStatus: false,
};

/**
 * @callback RenderView - custom render manager layout
 * @param {string} label
 * @param {Object[]} managerList - array of managers view: StaffCard
 * @return {React.element} - rendered view
 */

export default injectIntl(ManagersRow);
