import React from 'react';
import PropTypes from 'prop-types';
import { FormattedDate, FormattedMessage } from 'react-intl';
import isUndefined from 'lodash/isUndefined';
import Field from './Field';
import cssCompact from './compact.modules.scss';
import cssDefault from './default.modules.scss';

const LABEL = {
  COMPLEXITY: 'Сложность',
  STANDARD_HOURS: 'Кол-во часов',
  START_BEFORE_DATE: 'Крайний срок начала работы',
  DEADLINE_DATE: 'Расчетный срок выполнения',
  SPENT_HOURS: 'Затрачено',
};

const THEME = {
  DEFAULT: 'DEFAULT',
  COMPACT: 'COMPACT',
};

const CSS_THEME = {
  DEFAULT: cssDefault,
  COMPACT: cssCompact,
};

const previewPropTypes = { value: PropTypes.oneOfType([PropTypes.number, PropTypes.string]) };
const preivewDefaultProps = { value: undefined };

const MediaplannersWidget = props => {
  const {
    deadlineDate,
    complexity,
    standardHours,
    spentHours,
    startBeforeDate,
    canViewStandardHours,
    canViewSpentHours,
    canViewStartBeforeDate,
    canViewDeadlineDate,
    canViewComplexity,
    onSubmit,
    onSubmitFail,
    objectId,
    objectType,
    canEdit,
    canEditSpentHours,
    theme,
  } = props;

  const formName = `${objectType}:${objectId}:`;

  if (
    !(
      canViewStandardHours ||
      canViewSpentHours ||
      canViewStartBeforeDate ||
      canViewDeadlineDate ||
      canViewComplexity
    )
  ) {
    return null;
  }

  const css = CSS_THEME[theme];
  const isDefault = theme === THEME.DEFAULT;

  const Complexity = ({ value }) => (
    <span className={css.value}>
      <i className="fa fa-signal" aria-hidden="true" />
      {isUndefined(value) ? <span>&#8212;</span> : value}
    </span>
  );

  Complexity.propTypes = previewPropTypes;
  Complexity.defaultProps = preivewDefaultProps;

  const StandardHours = ({ value }) => (
    <span className={css.value}>
      <i className="fa fa-clock-o" aria-hidden="true" />
      {isUndefined(value) ? (
        <span>&#8212;</span>
      ) : (
        <FormattedMessage id="ID_HOUR" values={{ hours: value }} />
      )}
    </span>
  );

  StandardHours.propTypes = previewPropTypes;
  StandardHours.defaultProps = preivewDefaultProps;

  const SpentHours = ({ value }) => (
    <span className={css.value}>
      <i className="fa fa-hourglass-3" aria-hidden="true" />
      {isUndefined(value) ? (
        <span>&#8212;</span>
      ) : (
        <FormattedMessage id="ID_HOUR" values={{ hours: value }} />
      )}
    </span>
  );

  SpentHours.propTypes = previewPropTypes;
  SpentHours.defaultProps = preivewDefaultProps;

  const StartBeforeDate = ({ value }) => (
    <span className={css.value}>
      <i className="fa fa-play" aria-hidden="true" />
      {isUndefined(value) ? <span>&#8212;</span> : <FormattedDate value={value} />}
    </span>
  );

  StartBeforeDate.propTypes = previewPropTypes;
  StartBeforeDate.defaultProps = preivewDefaultProps;

  const DeadlineDate = ({ value }) => (
    <span className={css.value}>
      <i className="fa fa-calendar" aria-hidden="true" />
      {isUndefined(value) ? <span>&#8212;</span> : <FormattedDate value={value} />}
    </span>
  );

  DeadlineDate.propTypes = previewPropTypes;
  DeadlineDate.defaultProps = preivewDefaultProps;

  return (
    <div className={css.root}>
      <Field
        form={`${formName}complexity`}
        canView={canViewComplexity}
        canEdit={canEdit}
        isLabel={isDefault}
        name="complexity"
        theme={css}
        label={LABEL.COMPLEXITY}
        formType="NUMBER"
        value={complexity}
        preview={Complexity}
        onSubmit={onSubmit}
        onSubmitFail={onSubmitFail}
      />
      <Field
        form={`${formName}standardHours`}
        canView={canViewStandardHours}
        canEdit={canEdit}
        isLabel={isDefault}
        name="standardHours"
        theme={css}
        label={LABEL.STANDARD_HOURS}
        formType="NUMBER"
        value={standardHours}
        preview={StandardHours}
        onSubmit={onSubmit}
        onSubmitFail={onSubmitFail}
      />
      <Field
        form={`${formName}startBeforeDate`}
        canView={canViewStartBeforeDate}
        canEdit={canEdit}
        isLabel={isDefault}
        name="startBeforeDate"
        theme={css}
        label={LABEL.START_BEFORE_DATE}
        formType="DATE"
        value={startBeforeDate}
        preview={StartBeforeDate}
        onSubmit={onSubmit}
        onSubmitFail={onSubmitFail}
      />
      <Field
        form={`${formName}deadlineDate`}
        canView={canViewDeadlineDate}
        canEdit={canEdit}
        isLabel={isDefault}
        name="deadlineDate"
        theme={css}
        label={LABEL.DEADLINE_DATE}
        formType="DATE"
        value={deadlineDate}
        preview={DeadlineDate}
        onSubmit={onSubmit}
        onSubmitFail={onSubmitFail}
      />
      {isDefault && (
        <Field
          form={`${formName}spentHours`}
          canView={canViewSpentHours}
          canEdit={canEditSpentHours}
          isLabel={isDefault}
          name="spentHours"
          theme={css}
          label={LABEL.SPENT_HOURS}
          formType="NUMBER"
          value={spentHours}
          preview={SpentHours}
          onSubmit={onSubmit}
          onSubmitFail={onSubmitFail}
          isFloat
        />
      )}
    </div>
  );
};

MediaplannersWidget.propTypes = {
  deadlineDate: PropTypes.string,
  complexity: PropTypes.number,
  standardHours: PropTypes.number,
  spentHours: PropTypes.number,
  startBeforeDate: PropTypes.string,
  canViewStandardHours: PropTypes.bool,
  canViewSpentHours: PropTypes.bool,
  canViewStartBeforeDate: PropTypes.bool,
  canViewDeadlineDate: PropTypes.bool,
  canViewComplexity: PropTypes.bool,
  canEdit: PropTypes.bool,
  canEditSpentHours: PropTypes.bool,
  theme: PropTypes.string,
  objectType: PropTypes.oneOf(['request', 'ticket']),
  objectId: PropTypes.number,
  onSubmit: PropTypes.func.isRequired,
  onSubmitFail: PropTypes.func.isRequired,
};

MediaplannersWidget.defaultProps = {
  deadlineDate: undefined,
  complexity: undefined,
  standardHours: undefined,
  spentHours: undefined,
  startBeforeDate: undefined,
  objectId: undefined,
  objectType: undefined,
  canViewStandardHours: false,
  canViewSpentHours: false,
  canViewStartBeforeDate: false,
  canViewDeadlineDate: false,
  canViewComplexity: false,
  canEdit: false,
  canEditSpentHours: false,
  theme: THEME.DEFAULT,
};

MediaplannersWidget.THEME = THEME;

export default MediaplannersWidget;
