import React from 'react';
import PropTypes from 'prop-types';
import NumberForm from 'forms/Number';
import DateForm from 'forms/Date';
import Popup from '@crm/components/dist/lego2/Popup';
import Button from '@crm/components/dist/lego2/Button';

const FORM_TYPE = {
  NUMBER: NumberForm,
  DATE: DateForm,
};

const FORM_PROPS = {
  NUMBER: {
    maxLength: 4,
  },
};

export default class Field extends React.Component {
  static propTypes = {
    canEdit: PropTypes.bool,
    isLabel: PropTypes.bool,
    canView: PropTypes.bool,
    label: PropTypes.string,
    value: PropTypes.oneOfType([PropTypes.string, PropTypes.number]),
    name: PropTypes.string.isRequired,
    formType: PropTypes.oneOf(['NUMBER', 'DATE']).isRequired,
    preview: PropTypes.func.isRequired,
    theme: PropTypes.objectOf(PropTypes.string).isRequired,
  };

  static defaultProps = {
    canEdit: false,
    isLabel: false,
    canView: false,
    label: undefined,
    value: undefined,
  };

  constructor(props) {
    super(props);

    this.state = {
      showPopup: false,
    };
  }

  buttonRef = React.createRef();

  handleToggle = (flag) => {
    this.setState({ showPopup: typeof flag === 'boolean' ? flag : !this.state.showPopup });
  };

  render() {
    const Preview = this.props.preview;
    const css = this.props.theme;

    const renderEdit = () => {
      const passThroughtProps = { ...this.props };

      const initialValues = { [this.props.name]: this.props.value || '' };
      const Form = FORM_TYPE[this.props.formType];

      return (
        <React.Fragment>
          <Button size="xs" onClick={this.handleToggle} view="pseudo" ref={this.buttonRef}>
            <Preview value={this.props.value} />
          </Button>
          <Popup
            padding="xs"
            visible={this.state.showPopup}
            onOutsideClick={this.handleToggle}
            target="anchor"
            anchor={this.buttonRef}
          >
            <Form
              {...FORM_PROPS[this.props.formType]}
              initialValues={initialValues}
              onSubmitSuccess={() => this.handleToggle(false)}
              name={this.props.name}
              {...passThroughtProps}
            />
          </Popup>
        </React.Fragment>
      );
    };

    if (!this.props.canView) {
      return null;
    }

    return (
      <span className={css.item} title={!this.props.isLabel ? this.props.label : null}>
        {this.props.isLabel ? <span className={css.label}>{this.props.label}</span> : null}
        {this.props.canEdit ? renderEdit() : <Preview value={this.props.value} />}
      </span>
    );
  }
}
