import React, { MutableRefObject, createRef } from 'react';
import { FormSubscription } from 'final-form';
import { Form, Field, FormRenderProps } from 'react-final-form';
import TextInput from 'lego/final-form/TextInput';
import RichTextEditor from 'lego/final-form/RichTextEditor';
import AttachFiles from 'lego/final-form/AttachFiles';
import EmailInput from 'lego/final-form/EmailInput';
import Select from 'lego/final-form/Select';
import arrayMutators from 'final-form-arrays';
import memoize from 'memoize-one';
import Templates from 'modules/templates/components/Templates/Templates';
import PureSubjectDropdown, { asFinalFormField } from './SubjectDropdown';
import AutoSaveDate from './AutoSaveDate/AutoSaveDate';
import Tags from './Tags/Tags';
import Signature from './Signature';
import Row from './Row/Row';
import DelayDateControl from './DelayDateControl/DelayDateControl';
import SendButton from './SendButton/SendButton';
import Footer from './Footer/Footer';
import css from './NewMailForm.module.css';
import { FormValues, NewMailFormProps, NewMailFormApi } from './NewMailForm.types';

const SubjectDropdown = asFinalFormField(PureSubjectDropdown);

class NewMailForm extends React.Component<NewMailFormProps> {
  static defaultProps = {
    objectName: 'mail',
    autoFocusEditor: true,
    submitView: 'action',
    EmailInput,
  };

  private formMutators = {
    ...arrayMutators,
  };

  private formRef = createRef<HTMLFormElement>();

  private formSubscription: FormSubscription = { submitting: true };

  private formatFromForSelect = memoize((froms: string[]) =>
    froms.map((from) => ({ value: from, content: from })),
  );

  private renderForm: (props: FormRenderProps<FormValues>) => React.ReactNode = ({
    handleSubmit,
    form,
    submitting,
  }) => {
    const {
      objId,
      objectName,
      froms = [],
      subjects,
      signatures,
      onSignatureChange,
      templates,
      onTemplateClick,
      ckeditorRef,
      fileInputRef,
      formApiRef,
      autoSaveDate,
      addonAfter,
      addonBefore,
      addonBeforeActions,
      addonAfterActions,
      hasDelayedSend,
      autoFocusEditor,
      EmailInput,
      OpportunitiesInput,
      submitView,
    } = this.props;

    if (formApiRef) {
      (formApiRef as MutableRefObject<NewMailFormApi>).current = form;
    }

    const hasPredefinedSubjects = Boolean(Array.isArray(subjects) && subjects.length);

    return (
      <form onSubmit={handleSubmit} ref={this.formRef} className={css.NewMailForm}>
        {addonBefore}
        <Row label="От кого" multifield>
          <Field
            name="from"
            component={Select}
            placeholder="От кого"
            options={this.formatFromForSelect(froms)}
            hasEmptyValue={false}
            width="max"
          />
          <Signature signatures={signatures} onChange={onSignatureChange} />
          <Templates
            templates={templates}
            onTemplateClick={onTemplateClick}
            container={this.formRef}
          />
        </Row>
        <Row label="Кому">
          <Field name="to" component={EmailInput} placeholder="Кому" />
        </Row>
        <Row label="Копия">
          <Field name="cc" component={EmailInput} placeholder="Копия" />
        </Row>
        <Row label="Скрытая копия">
          <Field name="bcc" component={EmailInput} placeholder="Скрытая копия" />
        </Row>
        <Row label="Тема" multifield={hasPredefinedSubjects}>
          <Field name="subject" component={TextInput} placeholder="Тема" />
          <Field name="subject" component={SubjectDropdown} subjects={subjects} />
        </Row>
        {OpportunitiesInput && <OpportunitiesInput />}
        <Row>
          <Field name="tags" render={Tags} />
        </Row>
        <Row>
          <Field name="mailId" subscription={{ value: true }}>
            {(props) => {
              return (
                <Field
                  name="files"
                  component={AttachFiles}
                  objId={objId || props.input.value /* https://st.yandex-team.ru/CRM-11794 */}
                  objectName={objectName}
                  fileInputRef={fileInputRef}
                />
              );
            }}
          </Field>
        </Row>
        <Field
          className={css.NewMailForm__Body}
          name="body"
          component={RichTextEditor}
          ckeditorRef={ckeditorRef}
          focus={autoFocusEditor}
          resizable
        />
        <Footer className={css.NewMailForm__Footer}>
          <Footer.Left className={css.NewMailForm__Actions}>
            {addonBeforeActions}
            <SendButton type="submit" view={submitView} tabIndex={-1} disabled={submitting}>
              Отправить
            </SendButton>
            {hasDelayedSend && <DelayDateControl />}
            {addonAfterActions}
          </Footer.Left>
          {autoSaveDate && (
            <Footer.Right>
              <AutoSaveDate date={autoSaveDate} />
            </Footer.Right>
          )}
        </Footer>
        {addonAfter}
      </form>
    );
  };

  public render() {
    const { objId, objectName, ...props } = this.props;

    return (
      <Form
        {...props}
        render={this.renderForm}
        mutators={this.formMutators}
        subscription={this.formSubscription}
      />
    );
  }
}

export default NewMailForm;
