import React, { RefObject } from 'react';
import { MergeRef } from 'utils/mergeRef';
import { FormApi } from 'final-form';
import { CustomFormApi } from './withCustomSubmit';
import { FormValues } from '../NewMailForm.types';
import { AUTOSAVE_DELAY } from '../NewMailForm.config';

export interface WithAutoSaveProps {
  customFormApiRef?: RefObject<CustomFormApi>;
  hasAutoSave?: boolean;
  autoSaveOnUnmount?: boolean;
}

export interface WithAutoSaveWrappedProps {
  formApiRef?: RefObject<FormApi<FormValues>>;
  customFormApiRef?: RefObject<CustomFormApi>;
  autoSaveDate?: Date;
}

interface WithAutoSaveState {
  autoSaveDate?: Date;
}

export const withAutoSave = <T extends WithAutoSaveWrappedProps>(
  WrappedComponent: React.ComponentType<T>,
) => {
  class WithAutoSave extends React.Component<WithAutoSaveProps & T, WithAutoSaveState> {
    private formApiRef = new MergeRef<FormApi<unknown>>();

    private customFormApiRef = new MergeRef<CustomFormApi>();

    private timer?: number;

    public constructor(props) {
      super(props);

      this.state = {};
    }

    public componentDidMount(): void {
      if (this.props.hasAutoSave) {
        this.timer = window.setInterval(this.runAutoSave, AUTOSAVE_DELAY);
      }
    }

    public componentWillUnmount(): void {
      this.destroy();

      if (this.props.autoSaveOnUnmount && !this.isMailFormPristine()) {
        this.autoSaveOnUnmount();
      }
    }

    private autoSaveOnUnmount() {
      this.customFormApiRef.current!.submit({
        action: 'autoSave',
        cancelable: false,
      });
    }

    private destroy() {
      window.clearInterval(this.timer);
    }

    private submitSuccess = () => {
      this.setState({ autoSaveDate: new Date() });
    };

    private runAutoSave = async () => {
      if (this.isMailFormPristine()) {
        return;
      }

      if (this.customFormApiRef.current) {
        this.customFormApiRef.current.submit({
          action: 'autoSave',
          onSubmitSuccess: this.submitSuccess,
        });
      }
    };

    private isMailFormPristine() {
      return this.formApiRef.current!.getState().pristine;
    }

    public render() {
      const { formApiRef, customFormApiRef } = this.props;
      const { autoSaveDate } = this.state;

      this.formApiRef.refs = [formApiRef];

      this.customFormApiRef.refs = [customFormApiRef];

      return (
        <WrappedComponent
          {...this.props}
          autoSaveDate={autoSaveDate}
          customFormApiRef={this.customFormApiRef}
          formApiRef={this.formApiRef}
        />
      );
    }
  }

  return WithAutoSave;
};
