import React, { RefObject, MutableRefObject } from 'react';
import Bluebird from 'bluebird';
import { FormApi, Config } from 'final-form';
import { MergeRef } from 'utils/mergeRef';
import isPromise from 'utils/isPromise';
import { FormValues } from '../NewMailForm.types';
import { isMailIdResponse } from '../utils/isMailIdResponse';

export interface CustomSubmitProps {
  action?: string;
  cancelable?: boolean;
  onSubmitSuccess?: () => void;
}

export interface CustomFormApi {
  submit: (props: CustomSubmitProps) => Promise<FormValues | undefined> | undefined;
}

export interface WithCustomSubmitProps {
  formApiRef?: RefObject<FormApi<FormValues>>;
  onSubmit: Config['onSubmit'];
}

export interface WithCustomSubmitWrappedProps {
  formApiRef?: RefObject<FormApi<FormValues>>;
  customFormApiRef?: RefObject<CustomFormApi>;
  addonAfterActions?: React.ReactNode;
}

export const withCustomSubmit = <T extends WithCustomSubmitWrappedProps>(
  WrappedComponent: React.ComponentType<T>,
) => {
  class WithCustomSubmit extends React.Component<WithCustomSubmitProps & T> {
    public static defaultProps = {
      customSubmitPlaceholder: 'Custom submit',
    };

    private formApiRef = new MergeRef<FormApi<FormValues>>();

    private promise?: Bluebird<unknown> | Promise<unknown>;

    private currentSubmitProps?: CustomSubmitProps;

    public componentWillUnmount(): void {
      this.destroy();
    }

    private destroy() {
      if (
        this.currentSubmitProps &&
        this.currentSubmitProps.cancelable &&
        this.promise &&
        'cancel' in this.promise
      ) {
        this.promise.cancel();
      }
    }

    private runSubmit = (props: CustomSubmitProps) => {
      const formApi = this.formApiRef.current;

      if (!formApi) {
        return;
      }

      if (formApi.getState().submitting) {
        return;
      }

      this.setProps(props);
      return formApi.submit();
    };

    private updateMailId(data: unknown) {
      if (isMailIdResponse(data)) {
        this.formApiRef.current!.initialize((values) => {
          return {
            ...values,
            mailId: data.mailId,
          };
        });
      }
    }

    private handleSubmitSuccess = () => {
      if (this.currentSubmitProps && this.currentSubmitProps.onSubmitSuccess) {
        this.currentSubmitProps.onSubmitSuccess();
      }
    };

    // TODO: support callback arg
    private onSubmit: Config['onSubmit'] = (values, formApi) => {
      const result = this.props.onSubmit(
        { ...values, action: this.currentSubmitProps && this.currentSubmitProps.action },
        formApi,
      );
      if (isPromise(result)) {
        this.promise = result;
        (result as Promise<unknown>)
          .then((data) => {
            this.updateMailId(data);
            this.handleSubmitSuccess();
            return data;
          })
          .finally(() => {
            this.clearProps();
            this.promise = undefined;
          });
      } else {
        this.handleSubmitSuccess();
        this.clearProps();
      }

      return result;
    };

    private setProps(props?: CustomSubmitProps) {
      this.currentSubmitProps = { cancelable: true, ...props };
    }

    private clearProps() {
      this.currentSubmitProps = undefined;
    }

    public render() {
      const { formApiRef, customFormApiRef } = this.props;

      this.formApiRef.refs = [formApiRef];

      if (customFormApiRef) {
        (customFormApiRef as MutableRefObject<CustomFormApi>).current = { submit: this.runSubmit };
      }

      return (
        <WrappedComponent {...this.props} formApiRef={this.formApiRef} onSubmit={this.onSubmit} />
      );
    }
  }

  return WithCustomSubmit;
};
