import * as React from 'react';
import Bluebird from 'bluebird';
import isPromise from 'utils/isPromise';
import Spin from '@crm/components/dist/lego2/Spin';
import { AsyncRenderContext } from 'components/AsyncRender';

export interface WithLoadProps<LoadData> {
  onLoad: () => LoadData | Promise<LoadData> | Bluebird<LoadData>;
}

interface WithLoadState<LoadData> {
  data?: LoadData;
}

export const withLoad = <LoadData, T extends WithLoadProps<LoadData>>(
  WrappedComponent: React.ComponentType<T>,
) => {
  class WithLoad extends React.Component<T, WithLoadState<LoadData>> {
    static contextType = AsyncRenderContext;

    private promise?: Promise<LoadData> | Bluebird<LoadData>;

    public constructor(props) {
      super(props);

      this.state = {};
    }

    public componentDidMount(): void {
      const result = this.props.onLoad();
      if (isPromise(result)) {
        this.promise = result;
        this.context!.registerTask(this.promise);
        (result as Promise<LoadData>).then((data) => {
          this.setData(data);
          return data;
        });
      } else {
        this.setData(result);
      }
    }

    public componentWillUnmount(): void {
      if (this.promise && 'cancel' in this.promise) {
        this.promise.cancel();
      }

      if (this.promise) {
        this.context!.unregisterTask(this.promise);
      }
    }

    private setData(data: LoadData) {
      this.setState({ data });
    }

    public render() {
      const { data } = this.state;

      if (!data) {
        return (
          <div>
            <Spin view="default" size="l" progress />
          </div>
        );
      }

      return <WrappedComponent {...this.props} {...data} />;
    }
  }

  return WithLoad;
};
