import React, { RefObject } from 'react';
import { FormSpy } from 'react-final-form';
import { FormApi, Config } from 'final-form';
import { MergeRef } from 'utils/mergeRef';
import Select, { Option } from '@crm/components/dist/lego2/Select';
import { CustomFormApi } from './withCustomSubmit';

export interface WithCustomSubmitProps {
  customFormApiRef?: RefObject<CustomFormApi>;
  onSubmit: Config['onSubmit'];
  customSubmitOptions?: Option[];
  customSubmitPlaceholder?: string;
}

export interface WithCustomSubmitWrappedProps<FormValues = unknown> {
  customFormApiRef?: RefObject<FormApi<FormValues>>;
  addonBeforeActions?: React.ReactNode;
}

export const withSelectSubmit = <T extends WithCustomSubmitWrappedProps>(
  WrappedComponent: React.ComponentType<T>,
) => {
  class WithSelectSubmit extends React.Component<WithCustomSubmitProps & T> {
    public static defaultProps = {
      customSubmitPlaceholder: 'Custom submit',
    };

    private customFormApiRef = new MergeRef<CustomFormApi>();

    private onChange = (action: string) => {
      if (this.customFormApiRef.current) {
        this.customFormApiRef.current.submit({ action });
      }
    };

    public render() {
      const {
        addonBeforeActions,
        customSubmitOptions,
        customSubmitPlaceholder,
        customFormApiRef,
      } = this.props;

      this.customFormApiRef.refs = [customFormApiRef];

      return (
        <WrappedComponent
          {...this.props}
          customFormApiRef={this.customFormApiRef}
          addonBeforeActions={
            <>
              {addonBeforeActions}
              {customSubmitOptions && (
                <FormSpy subscription={{ submitting: true, hasValidationErrors: true }}>
                  {({ submitting, hasValidationErrors }) => (
                    <Select
                      disabled={submitting || hasValidationErrors}
                      hasEmptyValue={false}
                      placeholder={customSubmitPlaceholder}
                      showAlwaysPlaceholder
                      options={customSubmitOptions as Option[]}
                      onChange={this.onChange}
                      view="action"
                    />
                  )}
                </FormSpy>
              )}
            </>
          }
        />
      );
    }
  }

  return WithSelectSubmit;
};
