import React, { RefObject } from 'react';
import { MergeRef } from 'utils/mergeRef';
import { Ckeditor } from '../NewMailForm.types';
import SignatureEditor, { Signature } from '../utils/SignatureEditor';

interface Data {
  body?: string;
}

export interface SignatureChange {
  onSignatureChange: (signatureId: number) => void;
  ckeditorRef?: RefObject<Ckeditor>;
}

export interface WithSignatureInitProps {
  signatures: Signature[];
  initialValues?: Data;
  type: string;
  ckeditorRef?: RefObject<Ckeditor>;
}

export const withSignatureInit = <T extends WithSignatureInitProps>(
  WrappedComponent: React.ComponentType<T & SignatureChange>,
) => {
  class SignatureInit extends React.Component<T> {
    private signatureEditor: SignatureEditor;

    private ckeditorRef = new MergeRef<Ckeditor>();

    public constructor(props) {
      super(props);
      this.signatureEditor = new SignatureEditor(props.signatures);
    }

    public onChange = (signatureId: number) => {
      const ckeditor = this.ckeditorRef.current;

      if (ckeditor) {
        const isHtml = ckeditor.mode !== 'source';

        const updatedContent = this.signatureEditor.updateSignature(
          ckeditor.getData(),
          ckeditor.mode !== 'source',
          signatureId,
        );

        if (updatedContent) {
          ckeditor.setData(updatedContent);
          return;
        }

        if (isHtml) {
          const signatureTemplate = this.signatureEditor.getSignatureHtmlTemplate(signatureId);

          ckeditor.insertHtml(signatureTemplate);

          if (signatureTemplate) {
            const range = ckeditor.createRange();
            range.moveToElementEditStart(range.root);
            ckeditor.getSelection().selectRanges([range]);
          }
        }
      }
    };

    private insertSignatureInInitialValues(): Data {
      const { signatures, initialValues } = this.props;

      const signatureEditor = new SignatureEditor(signatures);

      const updatedBody = signatureEditor.insertSignatureHtml(
        (initialValues && initialValues.body) || '',
        signatureEditor.defaultSignatureId,
      );

      return { ...initialValues, body: updatedBody };
    }

    public render() {
      const { initialValues, type } = this.props;

      let patchedInitialValues = initialValues;
      if (type !== 'draft') {
        patchedInitialValues = this.insertSignatureInInitialValues();
      }

      this.ckeditorRef.refs = [this.props.ckeditorRef];

      return (
        <WrappedComponent
          {...this.props}
          ckeditorRef={this.ckeditorRef}
          initialValues={patchedInitialValues}
          onSignatureChange={this.onChange}
        />
      );
    }
  }

  return SignatureInit;
};
