import React, { RefObject } from 'react';
import { MergeRef } from 'utils/mergeRef';
import { Template } from 'types/TemplateProvider';
import { Ckeditor, FileInput } from '../NewMailForm.types';

interface Data {
  body?: string;
}

export interface TemplateController {
  onTemplateClick: (template: Template) => void;
  ckeditorRef?: RefObject<Ckeditor>;
  fileInputRef?: RefObject<FileInput>;
}

export interface Props {
  initialValues?: Data;
  ckeditorRef?: RefObject<Ckeditor>;
  fileInputRef?: RefObject<FileInput>;
}

export const withTemplateController = <T extends Props>(
  WrappedComponent: React.ComponentType<T & TemplateController>,
) => {
  class WithTemplateController extends React.Component<T> {
    private ckeditorRef = new MergeRef<Ckeditor>();

    private fileInputRef = new MergeRef<FileInput>();

    private onTemplateClick = (template: Template) => {
      const ckeditor = this.ckeditorRef.current;

      if (ckeditor) {
        if (ckeditor.mode === 'source') {
          ckeditor.setData(template.bodyPlain + ckeditor.getData());
        } else {
          ckeditor.insertHtml(template.bodyHtml);
        }
      }

      if (this.fileInputRef.current && Array.isArray(template.files)) {
        const fileIds = template.files.map((file) => file.id);
        if (fileIds.length) {
          this.fileInputRef.current.updateFiles(fileIds);
        }
      }
    };

    public render() {
      this.ckeditorRef.refs = [this.props.ckeditorRef];
      this.fileInputRef.refs = [this.props.fileInputRef];

      return (
        <WrappedComponent
          {...this.props}
          ckeditorRef={this.ckeditorRef}
          fileInputRef={this.fileInputRef}
          onTemplateClick={this.onTemplateClick}
        />
      );
    }
  }

  return WithTemplateController;
};
