import React, { MutableRefObject, createRef } from 'react';
import { FormSubscription } from 'final-form';
import 'assets/ys-font/fonts.css';
import { Field, FormRenderProps } from 'react-final-form';
import TextInput from 'lego/final-form/TextInput';
import RichTextEditor from 'lego/final-form/RichTextEditor';
import EmailInput from 'lego/final-form/EmailInput';
import Select from 'lego/final-form/Select';
import arrayMutators from 'final-form-arrays';
import memoize from 'memoize-one';
import { Toolbar2 } from 'components/Toolbar2';
import PureSubjectDropdown, { asFinalFormField } from 'components/NewMailForm/SubjectDropdown';
import AutoSaveDate from 'components/NewMailForm/AutoSaveDate/AutoSaveDate';
import DelayDateControl from 'components/NewMailForm/DelayDateControl/DelayDateControl';
import SendButton from 'components/NewMailForm/SendButton/SendButton';
import { FormValues, NewMailFormApi } from 'components/NewMailForm/NewMailForm.types';
import ButtonToggle from 'lego/components/ButtonToggle';
import { TemplatesMenuProps, TemplatesMenuService } from 'components/TemplatesMenu';
import { MailTemplateProvider, MailTemplateGroup } from 'services/MailTemplateProvider';
import {
  AttachFilesDNDContainer,
  AttachFilesViewerField,
  AttachFilesButton,
  AttachFilesDND,
} from 'components/AttachFiles';
import { RollerWithPopup } from 'components/Roller';
import { FormWithLoading } from 'components/FinalForm/withLoadingMutator';
import { Templates } from './Templates';
import { NewMailForm2Props } from './NewMailForm2.types';
import css from './NewMailForm2.module.css';
import Row from './Row/Row';

const SubjectDropdown = asFinalFormField(PureSubjectDropdown);

const formatTemplate = (template) => {
  const newTemplate = { ...template };
  newTemplate.id = template.templateId || template.signatureId;
  delete newTemplate.templateId;
  delete newTemplate.signatureId;
  return newTemplate;
};

export class NewMailForm2 extends React.Component<NewMailForm2Props> {
  static defaultProps = {
    objectName: 'mail',
    autoFocusEditor: true,
    submitView: 'action',
    EmailInput,
  };

  private formMutators = {
    ...arrayMutators,
  };

  private formRef = createRef<HTMLFormElement>();

  private formSubscription: FormSubscription = { submitting: true, hasValidationErrors: true };

  private formatFromForSelect = memoize((froms: string[]) =>
    froms.map((from) => ({ value: from, content: from })),
  );

  private createTemplatesMenuService = memoize(
    (signatures, templates) =>
      new TemplatesMenuService(
        new MailTemplateProvider(
          [
            Array.isArray(templates) &&
              templates.length && {
                id: 'template',
                caption: 'Шаблоны',
                items: templates.map(formatTemplate),
              },
            Array.isArray(signatures) &&
              signatures.length && {
                id: 'signature',
                caption: 'Подписи',
                items: signatures.map(formatTemplate),
              },
          ].filter(Boolean) as MailTemplateGroup[],
        ),
      ),
  );

  private handleTemplateClick: TemplatesMenuProps['onTemplateClick'] = (value) => {
    const { onSignatureChange, onTemplateClick } = this.props;

    if (value.groupId === 'signature') {
      onSignatureChange(Number(value.id));
    } else {
      onTemplateClick(value);
    }
  };

  private removeCkeditorDNDZones = (editor) => {
    // TODO: https://st.yandex-team.ru/CRM-15959
    editor._pastefileDNDHover?.destroy();
  };

  private renderForm: (props: FormRenderProps<FormValues>) => React.ReactNode = ({
    handleSubmit,
    form,
    submitting,
    hasValidationErrors,
    errors,
  }) => {
    const {
      objId,
      objectName,
      froms = [],
      subjects,
      signatures,
      templates,
      ckeditorRef,
      fileInputRef,
      formApiRef,
      autoSaveDate,
      addonAfter,
      addonBefore,
      addonBeforeActions,
      addonAfterActions,
      hasDelayedSend,
      autoFocusEditor,
      EmailInput,
      OpportunitiesInput,
      submitView,
      showSecondHeaderFields,
      onShowSecondHeaderFields,
      addonToolbarBefore,
      addonToolbarAfter,
      popupScope,
    } = this.props;

    if (formApiRef) {
      (formApiRef as MutableRefObject<NewMailFormApi>).current = form;
    }

    const hasPredefinedSubjects = Boolean(Array.isArray(subjects) && subjects.length);

    return (
      <AttachFilesDNDContainer>
        <form onSubmit={handleSubmit} ref={this.formRef} className={css.NewMailForm}>
          {addonBefore}
          <div className={css.NewMailForm__header}>
            <Row label="Кому" multifield>
              <Field
                name="to"
                component={EmailInput}
                className={css.NewMailForm__textinput}
                size="xs"
              />
              <ButtonToggle
                open={showSecondHeaderFields}
                onClick={onShowSecondHeaderFields}
                size="xs"
                view="clear"
              >
                Копии, От кого
              </ButtonToggle>
            </Row>
            <Row label="Тема" multifield={hasPredefinedSubjects}>
              <Field
                name="subject"
                component={TextInput}
                size="xs"
                className={css.NewMailForm__textinput}
              />
              <Field name="subject" component={SubjectDropdown} subjects={subjects} size="xs" />
            </Row>
            <Row label="Копия" hidden={!showSecondHeaderFields}>
              <Field
                name="cc"
                component={EmailInput}
                className={css.NewMailForm__textinput}
                size="xs"
              />
            </Row>
            <Row label="Скрытая" hidden={!showSecondHeaderFields}>
              <Field
                name="bcc"
                component={EmailInput}
                className={css.NewMailForm__textinput}
                size="xs"
              />
            </Row>
            <Row label="От кого" hidden={!showSecondHeaderFields}>
              <Field
                name="from"
                component={Select}
                options={this.formatFromForSelect(froms)}
                hasEmptyValue={false}
                size="xs"
              />
            </Row>
            {OpportunitiesInput && <OpportunitiesInput RowComponent={Row} />}
          </div>
          <Field
            name="body"
            component={RichTextEditor}
            ckeditorRef={ckeditorRef}
            focus={autoFocusEditor}
            autogrow
            defaultHeight={150}
            textAreaPadding={false}
            onInstanceReady={this.removeCkeditorDNDZones}
          />
          <Row>
            <Field name="mailId" subscription={{ value: true }}>
              {(props) => {
                return (
                  <Field
                    name="files"
                    component={AttachFilesViewerField}
                    objId={objId || props.input.value /* https://st.yandex-team.ru/CRM-11794 */}
                    objectName={objectName}
                    fileInputRef={fileInputRef}
                  />
                );
              }}
            </Field>
          </Row>
          <Toolbar2
            className={css.NewMailForm__Footer}
            left={
              <>
                {addonToolbarBefore}
                <AttachFilesButton disabled={errors.loading} fileInputRef={fileInputRef} />
                <Templates
                  templatesMenuService={this.createTemplatesMenuService(signatures, templates)}
                  onTemplateClick={this.handleTemplateClick}
                  scope={popupScope || this.formRef}
                />
              </>
            }
            right={
              <>
                <RollerWithPopup
                  className={css.NewMailForm__FooterRoller}
                  rollerClassNames={{
                    containerClassName: css.NewMailForm__FooterRollerContainer,
                  }}
                  popupProps={{
                    containerClassName: css.NewMailForm__FooterRollerPopupContainer,
                    scope: 'inplace',
                    boundary: this.formRef,
                  }}
                >
                  {addonBeforeActions}
                  <SendButton
                    type="submit"
                    view={submitView}
                    tabIndex={-1}
                    disabled={submitting || hasValidationErrors}
                  >
                    Отправить
                  </SendButton>
                  {hasDelayedSend && <DelayDateControl />}
                  {addonAfterActions}
                </RollerWithPopup>
                {autoSaveDate && <AutoSaveDate date={autoSaveDate} />}
                {addonToolbarAfter}
              </>
            }
          />
          {addonAfter}
          <AttachFilesDND
            zones={[
              {
                text: 'Перетащите изображение сюда, чтобы добавить в текст',
                onDrop: (acceptedFiles, rejectedFiles, event) => {
                  ckeditorRef?.current?.fire('drop', { $: event, dropRange: { select: () => {} } });
                },
                className: css.NewMailForm__attachInline,
              },
              {
                text: 'Перетащите файлы сюда, чтобы добавить вложение',
                onDrop: (files, rejected, event) => {
                  if (event?.type === 'dropFileId')
                    fileInputRef?.current?.updateFiles(files as number[], []);
                  else fileInputRef?.current?.addFiles(files as File[]);
                },
                disabled: errors.loading,
              },
            ]}
          />
        </form>
      </AttachFilesDNDContainer>
    );
  };

  public render() {
    const { objId, objectName, ...props } = this.props;

    return (
      <FormWithLoading
        {...props}
        render={this.renderForm}
        mutators={this.formMutators}
        subscription={this.formSubscription}
      />
    );
  }
}
