import React, { useState, useEffect, useContext, useRef, useMemo } from 'react';
import cx from 'classnames';
import { Subject } from 'rxjs';
import { AsyncRenderContext } from 'components/AsyncRender';
import debounce from 'lodash/debounce';
import $ from 'jquery';
import css from './RichHtmlEditor.module.css';
import { EditorProps, Ckeditor } from './RichHtmlEditor.types';
import { plainTextToHtml } from './RichHtmlEditor.utils';

import './ckeditor';

export const RichHtmlEditor: React.FC<EditorProps> = ({
  value,
  defaultValue = '',
  startupMode,
  resizable,
  autogrow,
  defaultHeight = 300,
  name = 'editor',
  editorConfig = {},
  focus,
  onBlur,
  onChange,
  onMount,
  onInstanceReady,
  onFocus,
  resizeHash,
  caretStartOnFocus,
  textAreaPadding = true,
  keepLinesInPlainText,
  className,
  height,
  theme,
  ckeditorRef,
}) => {
  const wrap = useRef<HTMLDivElement>(null);
  const asyncRender = useContext(AsyncRenderContext);

  const [ckeditor, setCkeditor] = useState<Ckeditor>();

  const defaultText = useMemo(() => value || defaultValue, [value, defaultValue]);

  const getEditorData = () => {
    if (ckeditor) {
      const data = ckeditor.getData();
      return keepLinesInPlainText ? data : plainTextToHtml(data);
    }

    return null;
  };

  const handleChange = debounce(() => {
    const editorData = getEditorData();
    if (editorData !== null) {
      onChange?.(editorData);
    }
  }, 300);

  const handleBlur = () => {
    const editorData = getEditorData();
    if (editorData !== null) {
      onBlur?.(editorData);
    }
  };

  const handleFocus = (...args) => {
    if (ckeditor?.mode === 'wysiwyg' && caretStartOnFocus) {
      // для wysiwyg ставим курсор в начало при фокусе
      const range = ckeditor.createRange();
      range.moveToElementEditStart(range.root);
      ckeditor.getSelection().selectRanges([range]);
    }

    onFocus?.(...args);
  };

  const handleInput = (event) => {
    if (event.data.$.inputType === 'insertReplacementText') {
      handleChange();
    }
  };

  const resize = () => {
    ckeditor?.fire('triggerResize');
  };

  const createExtraConfig = () => {
    const extraConfig: Record<string, unknown> = {};
    if (startupMode) {
      extraConfig.startupMode = startupMode;
    }

    if (resizable) {
      extraConfig.resize_enabled = true;
      extraConfig.resize_dir = 'vertical';
      extraConfig.height = defaultHeight;
    }

    if (autogrow) {
      extraConfig.autoGrow_enable = true;
      extraConfig.removePlugins = 'resize,resizewithwindow';
    }

    return extraConfig;
  };

  useEffect(() => {
    const asyncRenderTask = new Subject();
    asyncRender.registerTask(asyncRenderTask);

    const editor = global.CKEDITOR.replace(
      name,
      Object.assign({}, editorConfig, createExtraConfig()),
    );

    editor.on('instanceReady', () => {
      setCkeditor(editor);
      asyncRenderTask.complete();
    });

    return () => {
      asyncRenderTask.complete();
      editor.destroy();
      handleChange.cancel();
    };
  }, []);

  useEffect(() => {
    if (ckeditor) {
      if (autogrow) {
        const contentsElement = ckeditor.ui.contentsElement.$;
        contentsElement.style.maxHeight = '300px';
        contentsElement.style.minHeight = '80px';
      }

      const $switchBtn = $('.cke_button__switchmode');
      $switchBtn.text('Текстовый режим');
      $switchBtn.closest('.cke_toolbar').css({ float: 'right' });

      ckeditor.on('mode', () => {
        ckeditor.fire('change');
      });

      if (focus) {
        ckeditor.focus();
      }

      ckeditor.on('focus', handleFocus);
      ckeditor.on('blur', handleBlur);
      ckeditor.on('change', handleChange);

      const editable = ckeditor.editable();
      editable.attachListener(
        editable,
        global.CKEDITOR.env.ie ? 'keypress' : 'input',
        handleInput,
        this,
        null,
        999,
      );

      onMount?.(ckeditor);
      onInstanceReady?.(ckeditor);
      if (ckeditorRef) {
        ckeditorRef.current = ckeditor;
      }
    }
  }, [ckeditor]);

  useEffect(() => {
    let requestAnimationFrameID: number;
    const resizeObserver = new ResizeObserver(() => {
      requestAnimationFrameID = requestAnimationFrame(resize);
    }) as ResizeObserver | undefined;

    if (wrap.current && !resizable) {
      resizeObserver?.observe(wrap.current);
    }

    return () => {
      cancelAnimationFrame(requestAnimationFrameID);
      resizeObserver?.disconnect();
    };
  }, [resizable, wrap.current]);

  useEffect(() => {
    ckeditor?.setData(defaultValue);
  }, [defaultValue]);

  useEffect(() => {
    resize();
  }, [resizeHash]);

  return (
    <div
      className={cx(css.b, className, css[`b_height_${height}`], css[`b_theme_${theme}`], {
        [css.b_resizable]: resizable,
        [css.b_padding]: textAreaPadding,
      })}
      ref={wrap}
    >
      <textarea
        hidden
        className={css.textarea}
        name={name}
        cols={100}
        rows={6}
        defaultValue={defaultText}
      />
    </div>
  );
};

export default RichHtmlEditor;
