import {
    DATA_CKE_ROLE,
    MAX_QUOTES_COUNT,
    MAX_QUOTES_SYMBOLS,
    PLACEHOLDER_ROLE,
    QUOTE_CONTAINER_CLASS,
    EXPAND_QUOTE_CLASS,
    EXPAND_ALL_QUOTES_CLASS
} from './constants';

import {
    renderQuoteToggler,
    unmountQuoteToggler
} from './renderQuoteToggler';

import {
    calcQuoteLevelsCount,
    detachNode,
    extractQuotes,
    extractText,
    replaceNode,
    replaceQuotePlaceholder
} from './helpers';

const HIGHEST_EVENT_PRIORITY = -100;

export default class CollapsedQuotes {

    editor = null;

    _skipNextQuotesInsertion = false;
    _subscriptions = [];

    quoteNode = null;
    togglers = [];
    togglerPlaceholder = document.createElement('div');
    togglerReplacer = document.createElement('div');

    constructor(editor) {
        this.editor = editor;

        this.togglerPlaceholder.setAttribute(DATA_CKE_ROLE, PLACEHOLDER_ROLE);
        // CKEditor в IE 11 не ставит закрывающий тэг при обработке пустого div
        this.togglerPlaceholder.innerHTML = '&nbsp;<br>&nbsp;';

        this._patchUpdateElement();

        this._addSubscription('getData', this._onGetData, undefined, undefined, HIGHEST_EVENT_PRIORITY);
        this._addSubscription('setData', this._onSetData, undefined, undefined, HIGHEST_EVENT_PRIORITY);
        this._addSubscription('destroy', this._onDestroy);
        this._addSubscription('beforeCommandExec', this._onBeforeCommandExec);
        this._addSubscription('beforeSetMode', this._onBeforeModeChange, undefined, undefined, HIGHEST_EVENT_PRIORITY);

        this.editor.setData(this.editor.getData(), () => this.editor.updateElement());
    }

    _patchUpdateElement() {
        const sourceUpdateElement = this.editor.updateElement;

        this.editor.updateElement = (...args) => {
            const result = sourceUpdateElement.apply(this.editor, args);

            if (this.quoteNode) {
                replaceQuotePlaceholder(this.editor.container.$, this.firstToggler);
                this.editor.container.$.addEventListener('click', this._onClickHandler);
            }

            return result;
        };
    }

    _onClickHandler = (event) => {
        let target = event.target;

        while (target) {
            if (target.classList && target.classList.contains(EXPAND_QUOTE_CLASS)) {
                this._expandQuote();
                break;
            } else if (target.classList && target.classList.contains(EXPAND_ALL_QUOTES_CLASS)) {
                this._expandAllQuotes();
                break;
            }

            target = target.parentNode;
        }
    };

    _addSubscription(eventName, ...args) {
        this._subscriptions.push(
            this.editor.on(eventName, ...args)
        );
    }

    _onGetData = ({ data }) => {
        const message = data.dataValue;

        if (message !== undefined && !this._skipNextQuotesInsertion) {
            data.dataValue = this._insertQuoteBackIntoMessage(message);
        }

        this._skipNextQuotesInsertion = false;
    };

    _onSetData = ({ data }) => {
        if (this.editor.mode !== 'source' && !this.firstToggler) {
            this.quoteNode = null;
            this._clearTogglers();

            const quotesData = this._replaceInitialQuotesInMessage(data.dataValue);

            data.dataValue = quotesData.message;
            this.firstToggler = quotesData.toggler;
        }
    }

    _onDestroy = () => {
        this._subscriptions.forEach(
            (subscription) => subscription.removeListener()
        );

        this._clearTogglers();
    }

    _replaceInitialQuotesInMessage(message) {
        this.togglerReplacer.innerHTML = message;
        let toggler;

        const quoteLevels = calcQuoteLevelsCount(this.togglerReplacer);

        if (quoteLevels > 0) {
            toggler = this._generateQuoteTogglerNode({
                totalCount: quoteLevels,
                level: 0
            });

            let quotesTextLength = 0;
            let quotesLeft = extractQuotes(this.togglerReplacer, toggler);

            for (let i = 0; i < Math.min(quoteLevels, MAX_QUOTES_COUNT) && quotesLeft; ++i) {

                const currentQuote = quotesLeft;
                const currentToggler = this._generateQuoteTogglerNode(
                    { totalCount: quoteLevels, level: i + 1 }
                );

                quotesLeft = extractQuotes(quotesLeft, currentToggler);

                const quoteText = extractText(currentQuote);
                const togglerText = extractText(currentToggler);

                quotesTextLength += quoteText.length - togglerText.length;

                if (quotesTextLength < MAX_QUOTES_SYMBOLS) {
                    replaceNode(toggler, currentQuote);
                    toggler = currentToggler;
                } else {
                    replaceNode(currentToggler, quotesLeft);
                    quotesLeft = currentQuote;
                    break;
                }
            }

            replaceNode(toggler, this.togglerPlaceholder);

            this.quoteNode = quotesLeft;

            message = this.togglerReplacer.innerHTML;
            detachNode(this.quoteNode);
        }

        this.togglerReplacer.innerHTML = '';
        return { message, toggler };
    }

    _insertQuote(quoteContainer, level) {
        if (!quoteContainer) {
            return;
        }

        /*
        if (Daria.UA.BrowserName === 'MSIE') {
            this.editor.focusManager.blur(true);
        }
        */

        const { node, toggler } = this._prepareQuoteNode(quoteContainer, level);
        const processedNode = this._processQuoteNode(node, toggler);
        this._insertQuoteIntoSnapshots(processedNode);

        this.editor.fire('lockSnapshot');
        replaceNode(quoteContainer, processedNode);
        this.editor.fire('change');
        this.editor.fire('unlockSnapshot');
    }

    _insertQuoteIntoSnapshots(node) {
        if (!this.editor.undoManager) {
            return;
        }

        this.editor.undoManager.snapshots.forEach((snapshot) => {

            this.togglerReplacer.innerHTML = snapshot.contents;

            const containers = this.togglerReplacer
                .getElementsByClassName(QUOTE_CONTAINER_CLASS);

            if (!containers.length) {
                return;
            }

            const container = containers[0];

            replaceNode(container, node);

            snapshot.contents = this.togglerReplacer.innerHTML;

            detachNode(node);
            this.togglerReplacer.innerHTML = '';

            snapshot.bookmarks = [];
        });
    }

    _insertQuoteBackIntoMessage(message) {
        if (!this.quoteNode) {
            return message;
        }

        this.togglerReplacer.innerHTML = message;

        replaceNode(
            this.togglerReplacer.querySelector(`.${QUOTE_CONTAINER_CLASS}`),
            this.quoteNode
        );

        const result = this.togglerReplacer.innerHTML;

        detachNode(this.quoteNode);
        this.togglerReplacer.innerHTML = '';

        return result;
    }

    _processQuoteNode(node, toggler) {
        this.togglerReplacer.appendChild(node);

        const processedHtml = this.editor.dataProcessor
            .toHtml(this.togglerReplacer.innerHTML);

        detachNode(node);
        this.togglerReplacer.innerHTML = processedHtml;

        const processedNode = this.togglerReplacer.children[0];

        detachNode(processedNode);
        this.togglerReplacer.innerHTML = '';

        if (toggler) {
            replaceQuotePlaceholder(processedNode, toggler);
        }

        return processedNode;
    }

    _prepareQuoteNode(quoteContainer, level) {
        let node = this.quoteNode;
        let toggler;

        const showAll = level < 0;

        if (!showAll) {

            const quoteLevels = Number(quoteContainer.getAttribute('data-levels-count'))
                || calcQuoteLevelsCount(node);

            toggler = this._generateQuoteTogglerNode({
                totalCount: quoteLevels - 1,
                level: level + 1
            });

            const quoteNode = extractQuotes(node, this.togglerPlaceholder);

            this.quoteNode = quoteNode;

        } else {

            this.quoteNode = null;

        }

        return { node, toggler };
    }

    _getQuoteContainer() {
        if (!this.editor || !this.quoteNode) {
            return null;
        }

        const quoteContainer = this.editor.container.$
            .getElementsByClassName(QUOTE_CONTAINER_CLASS);

        if (!quoteContainer.length) {
            return null;
        }

        return quoteContainer[0];
    }

    _clearTogglers() {
        this.togglers.forEach(
            (toggler) => unmountQuoteToggler(toggler)
        );

        this.togglers = [];
    }

    _generateQuoteTogglerNode({ totalCount, level = 0 }) {
        if (totalCount <= 0) {
            return null;
        }

        const toggler = renderQuoteToggler({
            // lang: Daria.locale,
            level,
            onExpandAll: this._expandAllQuotes,
            onExpandQuote: this._expandQuote,
            totalCount
        });

        this.togglers.push(toggler);

        return toggler;
    }

    _onBeforeModeChange = ({ data }) => {
        if (data !== 'source') {
            return;
        }

        this._expandAllQuotes();
    };

    _onBeforeCommandExec = ({ data: { name } }) => {
        switch (name) {

            case 'translate_apply': {
                const translateData = this.editor._.translateData;
                const filledData = this._insertQuoteBackIntoMessage(translateData);
                this.editor._.translateData = filledData;
                break;
            }

            case 'translate':
                this._skipNextQuotesInsertion = true;
                break;
        }
    };

    _expandQuote = () => {
        const container = this._getQuoteContainer();

        if (!container) {
            return;
        }

        const level = Number(container.getAttribute('data-level'));

        this._insertQuote(container, level);
    };

    _expandAllQuotes = () => {
        const container = this._getQuoteContainer();

        this._insertQuote(container, -1);

        this._clearTogglers();
    };
}
