import React, { useLayoutEffect, useRef, useState, useMemo } from 'react';
import cx from 'classnames';
import css from './Roller.module.css';
import { RollerProps } from './Roller.types';
import { flatElement } from './Roller.utils';

export const Roller: React.FC<RollerProps> = ({
  containerClassName,
  className,
  children,
  onHiddenComponentsChange,
  updateKey = children,
  control,
}) => {
  const ref = useRef<HTMLDivElement>(null);

  const [height, setHeight] = useState<number>();
  const [parentBounding, setParentBounding] = useState<DOMRect>();
  const [disabled, setDisabled] = useState<boolean>(false);

  const flattenChildren = useMemo(() => React.Children.map(children, flatElement), [children]);

  const getContentWidth = () => {
    if (!ref.current) return 0;

    const { firstElementChild, lastElementChild } = ref.current;

    const leftOffset = firstElementChild?.getBoundingClientRect().left || 0;
    const rightOffset = lastElementChild?.getBoundingClientRect().right || 0;

    return rightOffset - leftOffset;
  };

  const setAllVisible = () => {
    ref.current?.childNodes.forEach((child: HTMLDivElement) => {
      child.style.display = 'unset';
    });
  };

  const getOutsideIndex = (width: number) => {
    if (!ref.current) return -1;

    const { firstElementChild, childNodes } = ref.current;
    const startX = firstElementChild?.getBoundingClientRect().x || 0;

    return Array.from(childNodes).findIndex((child: HTMLDivElement) => {
      const marginRight = parseInt(window.getComputedStyle(child).marginRight, 10) || 0;
      return child.getBoundingClientRect().right + marginRight - startX > width;
    });
  };

  const hideFromIndex = (fromIndex: number) => {
    ref.current?.childNodes.forEach((child: HTMLDivElement, index) => {
      if (index >= fromIndex) {
        child.style.display = 'none';
      }
    });
  };

  const splitHiddenComponents = () => {
    if (!ref.current) return;

    const parentBoundingWidth = ref.current.getBoundingClientRect().width;
    const outsideBoundingRight = parentBoundingWidth + (control?.visible ? control.width : 0);

    setAllVisible();

    let parentIndex = getOutsideIndex(outsideBoundingRight);

    if (parentIndex !== -1 && control) {
      parentIndex = getOutsideIndex(parentBoundingWidth - (control.visible ? 0 : control.width));
    }

    if (parentIndex !== -1) {
      hideFromIndex(parentIndex);
      onHiddenComponentsChange?.(flattenChildren?.slice(parentIndex));
    } else {
      onHiddenComponentsChange?.([]);
    }
  };

  useLayoutEffect(() => {
    if (ref.current) {
      const resizeObserver: ResizeObserver = new ResizeObserver(() => {
        setParentBounding(ref.current?.getBoundingClientRect());
      });

      resizeObserver.observe(ref.current);

      return () => {
        resizeObserver?.disconnect();
      };
    }
  }, []);

  useLayoutEffect(() => {
    setHeight(ref.current?.scrollHeight);
  }, [ref.current?.scrollHeight]);

  useLayoutEffect(() => {
    setAllVisible();
    setDisabled(getContentWidth() < (control?.width || 0));
  }, [updateKey, control?.width]);

  useLayoutEffect(() => {
    splitHiddenComponents();
  }, [updateKey, parentBounding, disabled, control?.width]);

  return (
    <div style={{ height }} className={cx(css.Roller, className)}>
      <div
        ref={ref}
        className={cx(
          { [css.Roller__container]: height && !disabled },
          css.Roller__container_flexRow,
          containerClassName,
        )}
      >
        {children}
      </div>
    </div>
  );
};
