import React, {
  useState,
  useRef,
  useLayoutEffect,
  ReactElement,
  ReactNodeArray,
  PropsWithChildren,
  Fragment,
} from 'react';
import cx from 'classnames';
import css from './RollerWithPopup.module.css';
import { Roller } from '../Roller';
import { RollerWithPopupProps } from './RollerWithPopup.types';
import { ButtonWithPopup } from './ButtonWithPopup';

export const RollerWithPopup = <T extends {}>({
  className,
  rollerClassNames,
  popupProps,
  moreButtonProps,
  children,
  onHiddenComponentsChange,
  updateKey,
}: PropsWithChildren<RollerWithPopupProps<T>>) => {
  const [hidden, setHidden] = useState<ReactElement[] | ReactNodeArray>([]);
  const [buttonWidth, setButtonWidth] = useState<number>(0);

  const buttonRef = useRef<HTMLDivElement>(null);

  const handleHiddenChange = (components: ReactElement[] | ReactNodeArray) => {
    setHidden(components);
    if (onHiddenComponentsChange) {
      onHiddenComponentsChange(components);
    }
  };

  useLayoutEffect(() => {
    if (buttonRef?.current) {
      setButtonWidth(buttonRef.current.getBoundingClientRect().width);
    }
  }, [buttonRef?.current?.offsetWidth]);

  return (
    <div className={cx(css.RollerWithPopup, className)}>
      <Roller
        {...rollerClassNames}
        updateKey={updateKey}
        onHiddenComponentsChange={handleHiddenChange}
        control={{
          width: buttonWidth,
          visible: hidden.length > 0,
        }}
      >
        {children}
      </Roller>
      {(hidden.length > 0 || !buttonWidth) && (
        <ButtonWithPopup
          onClick={moreButtonProps?.onClick}
          onOutsideClick={popupProps?.onOutsideClick}
          onVisibleChange={popupProps?.onVisibleChange}
          button={moreButtonProps?.element}
          popupProps={popupProps}
          innerRef={buttonRef}
        >
          <div className={cx(css.RollerWithPopup__popup, popupProps?.containerClassName)}>
            {hidden.map((item) => {
              const newElement = popupProps?.element?.(item.props);
              return (
                <Fragment key={item.key}>
                  {React.cloneElement(newElement || item, {
                    ...item.props,
                    ...newElement?.props,
                  })}
                </Fragment>
              );
            })}
          </div>
        </ButtonWithPopup>
      )}
    </div>
  );
};
