import React from 'react';
import PropTypes from 'prop-types';
import cx from 'classnames';
import get from 'lodash/get';

import cssClass from './style/style.scss';

const hasProps = (props) => (target) =>
  props.every((el) => typeof get(target.props, el) !== 'undefined');

const Spinner = ({
  visible,
  modal = true,
  full = false,
  delay,
  className = '',
  overlay = false,
  asComponent,
  children,
  zIndex = 10,
  inject,
  style,
}) => {
  const spinner = (modal) => (
    <div
      key="spinner"
      data-testid="spinner"
      className={cx(
        cssClass['crm-spinner'],
        cssClass[modal ? 'modal' : 'normal'],
        cssClass[modal ? 'spin' : 'spin2'],
        {
          [cssClass.spin2_size_l]: !modal,
          [cssClass.overlay__spinner]: overlay,
          [cssClass.spin2_progress_yes]: !modal,
          [cssClass['crm-hidden']]: !visible,
        },
      )}
    />
  );

  const renderOverlaySpinner = () =>
    (visible && (
      <div className={cx(cssClass.overlay__wrap, { [cssClass.delay]: delay })} style={{ zIndex }}>
        {spinner(false)}
      </div>
    )) ||
    null;

  if (overlay) {
    if (asComponent) {
      return renderOverlaySpinner();
    }

    if (inject) {
      const child = React.Children.only(children);

      return React.cloneElement(child, null, [child.props.children, renderOverlaySpinner()]);
    }

    return (
      <div className={cx(cssClass.overlay, className)} style={style}>
        {children}
        {renderOverlaySpinner()}
      </div>
    );
  }

  if (full) {
    return <div className={cssClass.wrap}>{spinner(false)}</div>;
  }

  return spinner(modal);
};

export const spinner = (props) => {
  return (target, property, descriptor) => {
    const origFunc = descriptor.value;

    descriptor.value = function __spinner() {
      if (!hasProps(props)(this)) {
        return (
          <div style={{ textAlign: 'center' }}>
            <Spinner visible modal={false} />
          </div>
        );
      }

      return origFunc.call(this);
    };
  };
};

Spinner.propTypes = {
  visible: PropTypes.bool,
  modal: PropTypes.bool,
  asComponent: PropTypes.bool,
  inject: PropTypes.bool,
  style: PropTypes.objectOf(PropTypes.any),
  delay: PropTypes.bool,
};

Spinner.defaultProps = {
  inject: false,
  style: undefined,
  delay: true,
};

export default Spinner;
