import React from 'react';
import { observer } from 'mobx-react-lite';
import Modal from '@crm/components/dist/lego2/Modal';
import Button from '@crm/components/dist/lego2/Button';
import Icon from '@crm/components/dist/lego2/Icon';
import { modal as createFormModal } from 'modules/issues/components/Forms/CreateForm';
import { OpportunityFormValues } from 'modules/issues/components/Forms/CreateForm/OpportunityForm/OpportunityForm.types';
import { ModuleName } from 'modules/issues/types';
import { useEditingService, StatefulProvider } from 'components/Table/hoc/withEdit/EditingContext';
import { TableEditable } from 'components/Table/TableEditable';
import { ThemeProvider } from 'components/Table/themes';
import { Overlay } from 'components/Overlay';
import {
  TableController,
  context as TableControllerContext,
} from 'components/Table/services/TableController';
import cx from 'classnames';
import { Text } from '@crm/components/dist/lego2/Text';
import { GridProps } from './ModalGrid.types';
import { mapIdsToSelectedItems, createNewOpportunity } from './ModalGrid.utils';
import css from './ModalGrid.module.css';

const ModalGridBase = observer((props: GridProps) => {
  const { visible, title, provider, selected = [], onClose, onSave, account } = props;
  const [isCreating, setIsCreating] = React.useState(false);

  const service = React.useMemo(() => new TableController().setPageSize(100000), []);
  const editingService = useEditingService();

  const selectedRows = React.useMemo(() => {
    if (editingService.selectedRows) {
      return new Set(editingService.selectedRows);
    }

    return new Set(selected.map((el) => String(el.id)));
  }, [selected, service.tableData]); // service.tableData нужна чтобы в таблице не слетали чекбоксы после загрузки данных

  React.useEffect(() => {
    if (visible) {
      service.setBaseUrl(provider).fetch();
    }
  }, [provider, visible]);

  const handleSubmit = () => {
    const ids = Array.from(editingService.selectedRows || []);
    onSave?.(mapIdsToSelectedItems(ids, service));
  };

  const handleOpportunityCreate = async (props: OpportunityFormValues) => {
    service.fetchingTableData = true;
    const id = await createNewOpportunity(props);
    await service.fetch();
    if (service.findRowById(id)) {
      editingService.selectRow(id);
    }
  };

  const onCreateOpportunityClick = () => {
    createFormModal.open(
      {
        onSubmit: (props: OpportunityFormValues) => {
          handleOpportunityCreate(props);
          setIsCreating(false);
        },
        moduleName: ModuleName.Opportunities,
        onClose: () => {
          setIsCreating(false);
        },
        account,
      },
      { size: 'small', hasAnimation: false },
    );
    setIsCreating(true);
  };

  const renderFooterLeft = (
    <Button
      size="m"
      iconLeft={(cls) => <Icon className={cls} svg="plus" />}
      onClick={onCreateOpportunityClick}
    >
      Создать сделку
    </Button>
  );

  return (
    <Modal
      className={cx({ [css.Modal_hidden]: isCreating })}
      visible={visible}
      hasAnimation={false}
      size="large"
      scrollableContent
      onClose={onClose}
      title={title || service.tableData?.meta.title}
      footerLeftChildren={renderFooterLeft}
      footerRightChildren={
        <>
          <Button size="m" onClick={onClose}>
            Отмена
          </Button>
          <Button
            size="m"
            onClick={handleSubmit}
            view="action"
            disabled={!service.tableData?.data?.length}
          >
            Готово
          </Button>
        </>
      }
    >
      <div className={css.Grid}>
        <Overlay
          display={service.fetchingTableData}
          hasSpinner
          zIndex={10}
          color="rgba(255,255,255,.9)"
        />

        {service.tableData?.data?.length ? (
          <TableControllerContext.Provider value={service}>
            <ThemeProvider value="mg">
              <TableEditable
                wrapSelfByContext={false}
                showEditPanel={false}
                tableData={service.tableData}
                selectedRows={selectedRows}
              />
            </ThemeProvider>
          </TableControllerContext.Provider>
        ) : (
          !service.fetchingTableData && (
            <Text
              as="div"
              align="center"
              typography="body-long-m"
              weight="medium"
              color="secondary"
              className={css.Grid__notFound}
            >
              У вас нет активных Сделок по Контрагенту или его Клиентам
            </Text>
          )
        )}
      </div>
    </Modal>
  );
});

export const ModalGrid = (props: GridProps) => {
  if (!props.visible) return null;

  return (
    <StatefulProvider>
      <ModalGridBase {...props} />
    </StatefulProvider>
  );
};
