import React, { useState, useRef, useMemo, useCallback } from 'react';
import Textinput from '@crm/components/dist/lego2/Textinput';
import { PopupMenu } from '@crm/components/dist/PopupMenu';
import Icon from '@crm/components/dist/lego2/Icon';
import { Bubble } from 'components/Bubble';
import { useSuggest } from './useSuggest';
import { mapToDictionary } from './SuggestWithTags.utils';
import { SuggestWithTagsProps } from './SuggestWithTags.types';
import css from './SuggestWithTags.module.css';

const Item = (item, { isSelected }) => {
  return (
    <div className={css.SuggestWithTags__item}>
      {item.name}
      {isSelected && <Icon className={css.SuggestWithTags__check} svg="check2" svgSize="m" />}
    </div>
  );
};

const BubbleItem = (item, { onDelete }) => {
  return (
    <Bubble canDelete onDelete={onDelete}>
      {item.name}
    </Bubble>
  );
};

const defaultMapper = (data) => {
  return data?.items ?? [];
};

export const SuggestWithTags: React.FC<SuggestWithTagsProps> = (props) => {
  const {
    renderItem = Item,
    renderBubble = BubbleItem,
    provider,
    mapper = defaultMapper,
    onChange,
    value = [],
    isSingleValue = false,
    iconRight,
    placeholder,
    clearSearchOnSelect = false,
  } = props;

  const inputRef = useRef<HTMLDivElement>(null);
  const [isEditing, setIsEditing] = useState(false);
  const [searchText, setSearchText] = useState('');

  const { items, isLoading } = useSuggest({
    text: searchText,
    url: provider,
    mapper,
    isEditing,
  });

  const itemsDict = useMemo(() => mapToDictionary(items), [items]);
  const valueDict = useMemo(() => mapToDictionary(value), [value]);

  const handleStartEditing = useCallback(() => {
    setIsEditing(true);
  }, []);

  const handleStopEditing = useCallback(() => {
    setIsEditing(false);
  }, []);

  const popupMenuItems = useMemo(() => {
    return items.map((item) => ({
      value: item.id,
      content: renderItem(item, {
        isSelected: Boolean(valueDict[item.id!]),
      }),
    }));
  }, [items, renderItem, valueDict]);

  const handleDelete = (id) => {
    onChange?.(value.filter((item) => item.id !== id));
    setIsEditing(false);
  };

  const handleChange = (id) => {
    if (clearSearchOnSelect) {
      setSearchText('');
    }

    if (valueDict[id]) {
      return handleDelete(id);
    }

    const item = itemsDict[id];
    onChange(isSingleValue ? [item] : [...value, item]);
    setIsEditing(false);
  };

  return (
    <div>
      {(!isSingleValue || !value?.length) && (
        <>
          <Textinput
            data-testid="SuggestTextInput"
            value={searchText}
            onChange={setSearchText}
            innerRef={inputRef}
            placeholder={placeholder}
            onFocus={handleStartEditing}
            iconRight={searchText ? undefined : iconRight}
          />
          <PopupMenu
            isLoading={isLoading}
            onClose={handleStopEditing}
            isVisible={isEditing}
            anchor={inputRef.current}
            items={popupMenuItems}
            onChange={handleChange}
          />
        </>
      )}
      {Boolean(value?.length) && (
        <div className={css.SuggestWithTags__bubbles}>
          {value.map((item) => renderBubble(item, { onDelete: () => handleDelete(item.id) }))}
        </div>
      )}
    </div>
  );
};
