import React, { useRef, useState, useEffect } from 'react';
import Icon from '@crm/components/dist/lego2/Icon';
import Popup from '@crm/components/dist/lego2/Popup';
import { Issue } from 'modules/issues';
import Button from '@crm/components/dist/lego2/Button';
import cx from 'classnames';
import { Text } from '@crm/components/dist/lego2/Text';
import Spin from '@crm/components/dist/lego2/Spin';
import Bluebird from 'bluebird';
import { getInternalIssueId, createInternalIssue } from './SupportChatIssue.utils';
import css from './SupportChatIssue.module.css';
import { SupportChatIssueProps, InternalIssueResponse } from './SupportChatIssue.types';

export const SupportChatIssue: React.FC<SupportChatIssueProps> = ({
  issueId,
  className,
  view = 'pseudo',
}) => {
  const [internalIssueId, setInternalIssueId] = useState<number>();
  const [error, setError] = useState<string>();
  const buttonRef = useRef(null);
  const [visible, setVisible] = useState<boolean>(false);
  const [isLoading, setLoading] = useState<boolean>(true);

  const togglePoppup = () => {
    setVisible(!visible);
  };

  const withRequestHandler = (request: Bluebird<InternalIssueResponse>) => {
    setLoading(true);
    request
      .then(({ issueId, error }) => {
        setError(error?.title);
        setInternalIssueId(issueId);
      })
      .catch((error: Error) => {
        setError(error.message);
      })
      .finally(() => {
        setLoading(false);
      });
  };

  const handleRequestClick = () => {
    withRequestHandler(createInternalIssue(issueId));
  };

  useEffect(() => {
    if (visible) {
      withRequestHandler(getInternalIssueId(issueId));
    } else {
      setInternalIssueId(undefined);
    }
  }, [visible]);

  const renderContent = () => {
    if (isLoading) {
      return (
        <div
          data-testid="spinner"
          className={cx(css.SupportChatIssue__issueContainer, css.SupportChatIssue_centered)}
        >
          <Spin view="default" position="center" progress size="m" />
        </div>
      );
    } else if (error) {
      return (
        <div className={cx(css.SupportChatIssue__errorContainer, css.SupportChatIssue_centered)}>
          <Text typography="body-long-m">{error}</Text>
        </div>
      );
    } else if (internalIssueId) {
      return (
        <div className={css.SupportChatIssue__issueContainer}>
          <Issue
            forceUseFeatureIssueTimelineV2
            isInChatsPopup
            id={internalIssueId}
            showAttributes={false}
            showTabsHeader={false}
            showHeader={false}
            showMailForm={false}
            showExpandWidgets={false}
            maxAccess={3}
          />
        </div>
      );
    }
    return (
      <div className={cx(css.SupportChatIssue__issueContainer, css.SupportChatIssue_centered)}>
        <Button onClick={handleRequestClick} size="m" view="action">
          Обратиться в 1.5 линию
        </Button>
      </div>
    );
  };

  return (
    <>
      <Button
        ref={buttonRef}
        view={view}
        size="xs"
        onClick={togglePoppup}
        icon={(cls) => <Icon className={cls} svg="internalChat" />}
      />
      <Popup
        hasTail
        visible={visible}
        target="anchor"
        anchor={buttonRef}
        className={className}
        direction={['bottom-start', 'bottom', 'bottom-end']}
        keepMounted={false}
        data-testid="popup-internal-chat"
      >
        <div className={css.SupportChatIssue__header}>
          <Button
            onClick={togglePoppup}
            size="s"
            view="clear"
            icon={(cls) => <Icon className={cls} svg="cross" />}
          />
        </div>
        {renderContent()}
      </Popup>
    </>
  );
};
