import React, { FC, useEffect, useRef, useState } from 'react';
import Button from '@crm/components/dist/lego2/Button';
import Icon from '@crm/components/dist/lego2/Icon';
import isEqual from 'lodash/isEqual';
import isEmpty from 'lodash/isEmpty';
import ToolTip from './Tooltip';
import css from './SupportChatWidget.module.css';
import { chatWidgetInit } from './utils/chatWidgetInit';
import { validationHelper } from './utils/validationHelper';
import { SupportChatWidgetProps } from './SupportChatWidget.types';
import { GUID_EMPTY_MESSAGE } from './SupportChatWidget.constants';
import { readLastSentMeta, writeLastSentMeta } from './utils/readWriteLastSentMeta';

let widget;

export const SupportChatWidget: FC<SupportChatWidgetProps> = (props) => {
  const popupTargetNode = useRef<HTMLDivElement>(null);
  const [errorMessage, setErrorMessage] = useState<string | null>(null);
  const [unreadCount, setUnreadCount] = useState(0);
  const [isWidgetInitialized, setWidgetInitialized] = useState(false);

  const sendServiceMessage = () => {
    const lastSentMeta = readLastSentMeta();
    if (props.meta && !isEmpty(props.meta)) {
      if (!isEqual(lastSentMeta, props.meta)) {
        widget.sendServiceMeta(props.meta);
        writeLastSentMeta(props.meta);
      }
    }
  };

  const handleOutsideClick = (event: MouseEvent) => {
    if (widget.isChatVisible) {
      const eventPath = event.composedPath();
      const isClickOnWidgetButton =
        popupTargetNode.current && eventPath.includes(popupTargetNode.current);
      if (!isClickOnWidgetButton) {
        widget.handleDocumentClick(event);
      }
    }
  };

  const handleUpdateUnreadCount = ({ value }) => {
    setUnreadCount(value);
  };

  const handleTooltipClose = () => {
    setErrorMessage(null);
  };

  const handleClick = (event) => {
    event.stopPropagation();
    event.preventDefault();

    if (errorMessage) {
      setErrorMessage(null);
      return;
    }

    if (widget.isChatVisible) {
      widget.hideChat();
      return;
    }

    if (!props.guid && props.validators === undefined) {
      setErrorMessage(GUID_EMPTY_MESSAGE);
      return;
    }

    const validationResult = props.validators
      ? props.validators.reduce(validationHelper, true)
      : true;

    if (validationResult === true && !widget.isChatVisible) {
      widget.showChat({
        guid: props.guid,
      });
      return;
    }

    if (typeof validationResult === 'string') {
      setErrorMessage(validationResult);
    }
  };
  useEffect(() => {
    try {
      chatWidgetInit(popupTargetNode.current);
      widget = window.chatWidget;
      setWidgetInitialized(true);
      document.addEventListener('click', handleOutsideClick, true);
      widget.addListener('chatShow', sendServiceMessage);
      widget.addListener('unreadCounterChange', handleUpdateUnreadCount);
      return () => {
        document.removeEventListener('click', handleOutsideClick, true);
        widget.removeListener('unreadCounterChange', handleUpdateUnreadCount);
        widget.removeListener('chatShow', sendServiceMessage);
      };
    } catch (e) {
      throw Error(e);
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [props.meta]);

  if (!isWidgetInitialized) {
    return <div ref={popupTargetNode} className={css.b} />;
  }

  const { fieldProps = {} } = props;

  return (
    <div ref={popupTargetNode} className={css.b}>
      {unreadCount > 0 && <div className={css.b__unreadCount} />}
      {errorMessage && (
        <ToolTip message={errorMessage} anchor={popupTargetNode} onClose={handleTooltipClose}>
          <div className={css.b__tooltipBody}>{errorMessage}</div>
        </ToolTip>
      )}
      <Button
        className={props.className}
        data-testid="show-widget-button"
        view="pseudo"
        size="xs"
        id="show-widget-button"
        onClick={handleClick}
        {...fieldProps}
      >
        <Icon svg="messengerOutline" />
      </Button>
    </div>
  );
};
