import React, { FC, useEffect, useMemo, useState } from 'react';
import cx from 'classnames';
import ClientExternalLinks, { Client } from 'components/ClientExternalLinks';
import { Text } from '@crm/components/dist/lego2/Text';
import { UserLink } from 'components/UserLink';
import * as keyset from 'pages/accounts/modules/info/modules/info/info.i18n';
import createI18N from '@yandex-int/i18n';
import Spin from '@crm/components/dist/lego2/Spin';
import { AccountShortInfo } from 'types/api/form/Fields/Account';
import css from './AccountTooltip.module.css';
import { AccountTooltipProps } from './AccountTooltip.types';
import { getAccountShortInfo } from '../getAccountShortInfo';

const i18n = createI18N(keyset);
const i18nContragent = i18n('contractors');
const i18nManagers = i18n('managers');
const i18nAgency = i18n('agency');

export const AccountTooltip: FC<AccountTooltipProps> = ({ id, login, type }) => {
  const [isLoading, setLoading] = useState(false);
  const [data, setData] = useState<AccountShortInfo>();
  const { services, info = {}, contragentName, agency, managers = [], tierName } = data ?? {};

  useEffect(() => {
    setLoading(true);
    getAccountShortInfo(id)
      .then(setData)
      .finally(() => setLoading(false));
  }, [id]);

  const externalLinksData = useMemo(() => {
    return {
      services,
      info: {
        id,
        login,
        ...info,
      },
    } as Client;
  }, [id, info, login, services]);

  if (isLoading) {
    return (
      <div className={css.AccountTooltip} data-testid="grid-cell-account-tooltip">
        <Spin view="default" position="center" progress size="xxs" />
      </div>
    );
  }

  return (
    <div className={css.AccountTooltip} data-testid="grid-cell-account-tooltip">
      <div className={css.AccountTooltip__block}>
        <ClientExternalLinks
          className={css.AccountTooltip__externalLinks}
          client={externalLinksData}
          tabIndex={-1}
        />
      </div>

      <div className={css.AccountTooltip__block}>
        <Text
          typography="body-long-s"
          color="secondary"
          data-testid="tier-label"
          as="div"
          className={cx({ [css.AccountTooltip__empty]: !tierName })}
        >
          Tier
        </Text>
        {tierName && (
          <Text typography="body-long-m" as="div">
            {tierName}
          </Text>
        )}
      </div>

      {type !== 'Contractor' && (
        <div className={css.AccountTooltip__block}>
          <Text
            typography="body-long-s"
            color="secondary"
            data-testid="contragent-label"
            as="div"
            className={cx({ [css.AccountTooltip__empty]: !contragentName })}
          >
            {i18nContragent}
          </Text>
          <Text typography="body-long-m" as="div">
            {contragentName}
          </Text>
        </div>
      )}

      <div className={css.AccountTooltip__block}>
        <Text
          typography="body-long-s"
          color="secondary"
          data-testid="managers-label"
          as="div"
          className={cx({ [css.AccountTooltip__empty]: !managers?.length })}
        >
          {i18nManagers}
        </Text>
        {managers.map((manager) => (
          <UserLink
            className={css.AccountTooltip__manager}
            user={manager}
            isTooltipDisabled
            showCrmPosition
            showStatus
            key={manager.id}
          />
        ))}
      </div>

      <div className={css.AccountTooltip__block}>
        <Text
          typography="body-long-s"
          color="secondary"
          data-testid="agency-label"
          className={cx({ [css.AccountTooltip__empty]: !agency })}
        >
          {i18nAgency}
        </Text>
        <Text typography="body-long-m" as="div">
          {agency}
        </Text>
      </div>
    </div>
  );
};
