import React, { FC, createContext, CSSProperties, useContext } from 'react';
import { observer } from 'mobx-react-lite';
import cx from 'classnames';
import Spin from '@crm/components/dist/lego2/Spin';
import { RowProps } from './Row.types';
import CellComponent from './Cell';
import { CheckboxForSelect } from '../hoc/withEdit/components/CheckboxForSelect';
import { useEditingService } from '../hoc/withEdit/EditingContext';
import themes from './themes';
import { themeContext } from '../themes';

export const RowIdContext = createContext<string>('');

const needPassOptions = new Set(['DateTime', 'CustomView']);

const Row: FC<RowProps> = observer(
  ({
    row,
    even,
    columns,
    isLoading,
    gridTemplate,
    withSelect = false,
    pinCheckbox = false,
    fieldsSettings,
  }) => {
    const theme = useContext(themeContext);
    const css = themes[theme] || themes.default;
    const style: CSSProperties = {
      gridTemplateColumns: gridTemplate,
    };

    const service = useEditingService();
    const isSelected = service.selectedRows?.has(String(row.id));

    return (
      <RowIdContext.Provider value={row.id}>
        <div
          className={cx(css.Row, { [css.Row__even]: even, [css.Row_selected]: isSelected })}
          style={style}
        >
          {isLoading && (
            <div className={css.Row__loadingOverlay}>
              <Spin view="default" size="xxs" progress />
            </div>
          )}
          {withSelect && (
            <CheckboxForSelect
              className={cx({
                [css.Row__checkboxCell_sticky]: pinCheckbox,
              })}
              id={row.id}
            />
          )}
          {columns.map((column) => {
            const cell = row.fields.find((dataField) => dataField.id === column.id);
            if (!cell) {
              return null;
            }
            const className = cx({
              [css.Row__cell_sticky]: fieldsSettings[column.id]?.isPinned ?? column.isPinned,
            });

            return (
              <CellComponent
                access={column.access}
                key={`${column.id}-${row.id}`}
                className={className}
                cell={cell}
                options={needPassOptions.has(column.type) && column.options}
              />
            );
          })}
        </div>
      </RowIdContext.Provider>
    );
  },
);

export default Row;
