import React, { FC, useContext } from 'react';
import cx from 'classnames';
import { Overlay } from 'components/Overlay';
import { EmptyTable } from './EmptyTable';
import { TableProps, Column } from './Table.types';
import { isRowLoading } from './utils/isRowLoading';
import { getGridTemplateColumns } from './utils/getGridTemplateColumns';
import Row from './Row';
import TableHeader from './TableHeader';
import { themes, themeContext } from './themes';
import { mapFieldsSettingsToDict } from './utils/mapFieldsSettingsToDict';

const Table: FC<TableProps> = ({
  tableData,
  className,
  rowsInLoading = [],
  withControls,
  selectedRows,
  bunkerSettings,
  fetchingData,
}) => {
  const theme = useContext(themeContext);
  const css = themes[theme] || themes.default;
  const withSelect = Boolean(selectedRows);
  const columns = tableData.meta.fieldsVisibility
    .map((fieldId) => tableData.meta.fields.find((field) => field.id === fieldId))
    .filter(Boolean) as Column[];

  const gridTemplate = getGridTemplateColumns(columns, {
    withSelect,
    theme,
    fieldsSettings: bunkerSettings?.fieldsSettings,
  });

  const fieldsSettings = bunkerSettings?.fieldsSettings
    ? mapFieldsSettingsToDict(bunkerSettings.fieldsSettings)
    : {};

  return (
    <div className={cx(css.Table, className)} data-testid="Table">
      <div className={css.Table__body}>
        <TableHeader
          withSelect={withSelect}
          pinCheckbox={bunkerSettings?.pinCheckbox}
          withControls={withControls}
          columns={columns}
          fieldsSettings={fieldsSettings}
          gridTemplate={gridTemplate}
        />
        <div className={css.Table__rowsSection}>
          <Overlay display={fetchingData} color="rgba(255,255,255,.7)" zIndex={10} />
          {!tableData?.data?.length ? (
            <EmptyTable />
          ) : (
            tableData.data.map((row, rowIndex) => (
              <Row
                gridTemplate={gridTemplate}
                isLoading={isRowLoading(rowsInLoading, row.id)}
                columns={columns}
                fieldsSettings={fieldsSettings}
                key={row.id}
                even={Boolean(rowIndex % 2)}
                row={row}
                withSelect={withSelect}
                pinCheckbox={bunkerSettings?.pinCheckbox}
              />
            ))
          )}
        </div>
      </div>
    </div>
  );
};

export default Table;
